/*
 * This file is part of RebornCore, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2021 TeamReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.multiblock;

import reborncore.RebornCore;
import reborncore.common.util.WorldUtils;

import java.util.HashSet;
import java.util.LinkedList;
import java.util.Set;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2487;
import net.minecraft.class_2586;
import net.minecraft.class_2818;

/**
 * This class contains the base logic for "multiblock controllers".
 * Conceptually, they are meta-TileEntities. They govern the logic for an
 * associated group of TileEntities.
 * <p>
 *  Subordinate TileEntities implement the {@link IMultiblockPart} class and, generally,
 *  should not have an update() loop.
 * </p>
 */
public abstract class MultiblockControllerBase {
	public static final short DIMENSION_UNBOUNDED = -1;

	// Multiblock stuff - do not mess with
	protected class_1937 worldObj;

	// Disassembled -> Assembled; Assembled -> Disassembled OR Paused; Paused ->
	// Assembled
	protected enum AssemblyState {
		Disassembled, Assembled, Paused
	}

	protected AssemblyState assemblyState;

	public HashSet<IMultiblockPart> connectedParts;

	/**
	 * This is a deterministically-picked coordinate that identifies this
	 * multiblock uniquely in its dimension. Currently, this is the coord with
	 * the lowest X, Y and Z coordinates, in that order of evaluation. i.e. If
	 * something has a lower X but higher Y/Z coordinates, it will still be the
	 * reference. If something has the same X but a lower Y coordinate, it will
	 * be the reference. Etc.
	 */
	private class_2338 referenceCoord;

	/**
	 * Minimum bounding box coordinate. Blocks do not necessarily exist at this
	 * coord if your machine is not a cube/rectangular prism.
	 */
	private class_2338 minimumCoord;

	/**
	 * Maximum bounding box coordinate. Blocks do not necessarily exist at this
	 * coord if your machine is not a cube/rectangular prism.
	 */
	private class_2338 maximumCoord;

	/**
	 * Set to true whenever a part is removed from this controller.
	 */
	private boolean shouldCheckForDisconnections;

	/**
	 * Set whenever we validate the multiblock
	 */
	private MultiblockValidationException lastValidationException;

	protected boolean debugMode;

	protected MultiblockControllerBase(class_1937 world) {
		// Multiblock stuff
		worldObj = world;
		connectedParts = new HashSet<>();

		referenceCoord = null;
		assemblyState = AssemblyState.Disassembled;

		minimumCoord = null;
		maximumCoord = null;

		shouldCheckForDisconnections = true;
		lastValidationException = null;

		debugMode = false;
	}

	public void setDebugMode(boolean active) {
		debugMode = active;
	}

	public boolean isDebugMode() {
		return debugMode;
	}

	/**
	 * Call when a block with cached save-delegate data is added to the
	 * multiblock. The part will be notified that the data has been used after
	 * this call completes.
	 *
	 * @param part {@link IMultiblockPart} Attached part
	 * @param data {@link class_2487} The NBT tag containing this controller's data.
	 */
	public abstract void onAttachedPartWithMultiblockData(IMultiblockPart part, class_2487 data);

	/**
	 * Check if a block is being tracked by this machine.
	 *
	 * @param blockCoord {@link class_2338} Coordinate to check.
	 * @return {@code boolean} True if the blockEntity entity at blockCoord is being tracked by this
	 * machine, false otherwise.
	 */
	public boolean hasBlock(class_2338 blockCoord) {
		return connectedParts.contains(blockCoord);
	}

	/**
	 * Attach a new part to this machine.
	 *
	 * @param part {@link IMultiblockPart} The part to add.
	 */
	public void attachBlock(IMultiblockPart part) {
		//IMultiblockPart candidate;
		class_2338 coord = part.getWorldLocation();

		if (!connectedParts.add(part)) {
			RebornCore.LOGGER.warn(
				String.format("[%s] Controller %s is double-adding part %d @ %s. This is unusual. If you encounter odd behavior, please tear down the machine and rebuild it.",
					(worldObj.method_8608() ? "CLIENT" : "SERVER"), hashCode(), part.hashCode(), coord));
		}

		part.onAttached(this);
		this.onBlockAdded(part);

		if (part.hasMultiblockSaveData()) {
			class_2487 savedData = part.getMultiblockSaveData();
			onAttachedPartWithMultiblockData(part, savedData);
			part.onMultiblockDataAssimilated();
		}

		if (this.referenceCoord == null) {
			referenceCoord = coord;
			part.becomeMultiblockSaveDelegate();
		} else if (coord.method_10265(referenceCoord) < 0) {
			class_2586 te = this.worldObj.method_8321(referenceCoord);
			((IMultiblockPart) te).forfeitMultiblockSaveDelegate();

			referenceCoord = coord;
			part.becomeMultiblockSaveDelegate();
		} else {
			part.forfeitMultiblockSaveDelegate();
		}

		boolean updateRequired = false;
		class_2338 partPos = part.method_11016();

		if (minimumCoord != null) {

			if (partPos.method_10263() < minimumCoord.method_10263()) {
				updateRequired = true;
			}
			if (partPos.method_10264() < minimumCoord.method_10264()) {
				updateRequired = true;
			}
			if (partPos.method_10260() < minimumCoord.method_10260()) {
				updateRequired = true;
			}
			if (updateRequired) {
				this.minimumCoord = new class_2338(partPos.method_10263(), partPos.method_10264(), partPos.method_10260());
			}
		}

		if (maximumCoord != null) {
			if (partPos.method_10263() > maximumCoord.method_10263()) {
				updateRequired = true;
			}
			if (partPos.method_10264() > maximumCoord.method_10264()) {
				updateRequired = true;
			}
			if (partPos.method_10260() > maximumCoord.method_10260()) {
				updateRequired = true;
			}
			if (updateRequired) {
				this.maximumCoord = new class_2338(partPos.method_10263(), partPos.method_10264(), partPos.method_10260());
			}
		}

		MultiblockRegistry.addDirtyController(worldObj, this);
	}

	/**
	 * Called when a new part is added to the machine. Good time to register
	 * things into lists.
	 *
	 * @param newPart {@link IMultiblockPart} The part being added.
	 */
	protected abstract void onBlockAdded(IMultiblockPart newPart);

	/**
	 * Called when a part is removed from the machine. Good time to clean up
	 * lists.
	 *
	 * @param oldPart {@link IMultiblockPart} The part being removed.
	 */
	protected abstract void onBlockRemoved(IMultiblockPart oldPart);

	/**
	 * Called when a machine is assembled from a disassembled state.
	 */
	protected abstract void onMachineAssembled();

	/**
	 * Called when a machine is restored to the assembled state from a paused
	 * state.
	 */
	protected abstract void onMachineRestored();

	/**
	 * Called when a machine is paused from an assembled state This generally
	 * only happens due to chunk-loads and other "system" events.
	 */
	protected abstract void onMachinePaused();

	/**
	 * Called when a machine is disassembled from an assembled state. This
	 * happens due to user or in-game actions (e.g. explosions)
	 */
	protected abstract void onMachineDisassembled();

	/**
	 * Callback whenever a part is removed (or will very shortly be removed)
	 * from a controller. Do housekeeping/callbacks, also nulls min/max coords.
	 *
	 * @param part {@link IMultiblockPart} The part being removed.
	 */
	private void onDetachBlock(IMultiblockPart part) {
		// Strip out this part
		part.onDetached(this);
		this.onBlockRemoved(part);
		part.forfeitMultiblockSaveDelegate();

		minimumCoord = maximumCoord = null;

		if (referenceCoord != null && referenceCoord.equals(part.method_11016())) {
			referenceCoord = null;
		}

		shouldCheckForDisconnections = true;
	}

	/**
	 * Call to detach a block from this machine. Generally, this should be
	 * called when the {@link class_2586} entity is being released, e.g. on block destruction.
	 *
	 * @param part           {@link IMultiblockPart} The part to detach from this machine.
	 * @param chunkUnloading {@code boolean} Is this entity detaching due to the chunk unloading? If true,
	 *                       the multiblock will be paused instead of broken.
	 */
	public void detachBlock(IMultiblockPart part, boolean chunkUnloading) {
		if (chunkUnloading && this.assemblyState == AssemblyState.Assembled) {
			this.assemblyState = AssemblyState.Paused;
			this.onMachinePaused();
		}

		// Strip out this part
		onDetachBlock(part);
		if (!connectedParts.remove(part)) {
			RebornCore.LOGGER.warn(
				String.format("[%s] Double-removing part (%d) @ %d, %d, %d, this is unexpected and may cause problems. If you encounter anomalies, please tear down the reactor and rebuild it.",
					worldObj.method_8608() ? "CLIENT" : "SERVER", part.hashCode(), part.method_11016().method_10263(),
					part.method_11016().method_10264(), part.method_11016().method_10260()));
		}

		if (connectedParts.isEmpty()) {
			// Destroy/unregister
			MultiblockRegistry.addDeadController(this.worldObj, this);
			return;
		}

		MultiblockRegistry.addDirtyController(this.worldObj, this);

		// Find new save delegate if we need to.
		if (referenceCoord == null) {
			selectNewReferenceCoord();
		}
	}

	/**
	 * Helper method so we don't check for a whole machine until we have enough
	 * blocks to actually assemble it. This isn't as simple as xmax*ymax*zmax
	 * for non-cubic machines or for machines with hollow/complex interiors.
	 *
	 * @return {@code int} The minimum number of blocks connected to the machine for it to
	 * be assembled.
	 */
	protected abstract int getMinimumNumberOfBlocksForAssembledMachine();

	/**
	 * Returns the maximum X dimension size of the machine, or -1
	 * (DIMENSION_UNBOUNDED) to disable dimension checking in X. (This is not
	 * recommended.)
	 *
	 * @return {@code int} The maximum X dimension size of the machine, or -1
	 */
	protected abstract int getMaximumXSize();

	/**
	 * Returns the maximum Z dimension size of the machine, or -1
	 * (DIMENSION_UNBOUNDED) to disable dimension checking in X. (This is not
	 * recommended.)
	 *
	 * @return {@code int} The maximum Z dimension size of the machine, or -1
	 */
	protected abstract int getMaximumZSize();

	/**
	 * Returns the maximum Y dimension size of the machine, or -1
	 * (DIMENSION_UNBOUNDED) to disable dimension checking in X. (This is not
	 * recommended.)
	 *
	 * @return {@code int} The maximum Y dimension size of the machine, or -1
	 */
	protected abstract int getMaximumYSize();

	/**
	 * Returns the minimum X dimension size of the machine. Must be at least 1,
	 * because nothing else makes sense.
	 *
	 * @return {@code int} The minimum X dimension size of the machine
	 */
	protected int getMinimumXSize() {
		return 1;
	}

	/**
	 * Returns the minimum Y dimension size of the machine. Must be at least 1,
	 * because nothing else makes sense.
	 *
	 * @return {@code int} The minimum Y dimension size of the machine
	 */
	protected int getMinimumYSize() {
		return 1;
	}

	/**
	 * Returns the minimum Z dimension size of the machine. Must be at least 1,
	 * because nothing else makes sense.
	 *
	 * @return {@code int} The minimum Z dimension size of the machine
	 */
	protected int getMinimumZSize() {
		return 1;
	}

	/**
	 * @return {@link MultiblockValidationException} An exception representing the last error encountered when trying
	 * to assemble this multiblock, or {@code null} if there is no error.
	 */
	public MultiblockValidationException getLastValidationException() {
		return lastValidationException;
	}

	/**
	 * Checks if a machine is whole. If not, throws an exception with the reason
	 * why.
	 *
	 * @throws MultiblockValidationException if the machine is not whole.
	 */
	protected abstract void isMachineWhole() throws MultiblockValidationException;

	/**
	 * Check if the machine is whole or not. If the machine was not whole, but
	 * now is, assemble the machine. If the machine was whole, but no longer is,
	 * disassemble the machine.
	 */
	public void checkIfMachineIsWhole() {
		AssemblyState oldState = this.assemblyState;
		boolean isWhole;
		this.lastValidationException = null;
		try {
			isMachineWhole();
			isWhole = true;
		} catch (MultiblockValidationException e) {
			lastValidationException = e;
			isWhole = false;
		}

		if (isWhole) {
			// This will alter assembly state
			assembleMachine(oldState);
		} else if (oldState == AssemblyState.Assembled) {
			// This will alter assembly state
			disassembleMachine();
		}
		// Else Paused, do nothing
	}

	/**
	 * Called when a machine becomes "whole" and should begin functioning as a
	 * game-logically finished machine. Calls {@link #onMachineAssembled} on all attached
	 * parts.
	 *
	 * @param oldState {@link AssemblyState} The previous state of the machine.
	 */
	private void assembleMachine(AssemblyState oldState) {
		for (IMultiblockPart part : connectedParts) {
			part.onMachineAssembled(this);
		}

		this.assemblyState = AssemblyState.Assembled;
		if (oldState == AssemblyState.Paused) {
			onMachineRestored();
		} else {
			onMachineAssembled();
		}
	}

	/**
	 * Called when the machine needs to be disassembled. It is not longer
	 * "whole" and should not be functional, usually as a result of a block
	 * being removed. Calls {@link IMultiblockPart#onMachineBroken} on all attached parts.
	 */
	private void disassembleMachine() {
		for (IMultiblockPart part : connectedParts) {
			part.onMachineBroken();
		}

		this.assemblyState = AssemblyState.Disassembled;
		onMachineDisassembled();
	}

	/**
	 * Assimilate another controller into this controller. Acquire all the
	 * other controller's blocks and attach them to this one.
	 *
	 * @param other {@link MultiblockControllerBase} The controller to merge into this one.
	 * @throws IllegalArgumentException if the controller with the lowest minimum-coord value does not consume
	 *                                  the one with the higher value.
	 */
	public void assimilate(MultiblockControllerBase other) {
		class_2338 otherReferenceCoord = other.getReferenceCoord();
		if (otherReferenceCoord != null && getReferenceCoord().method_10265(otherReferenceCoord) >= 0) {
			throw new IllegalArgumentException(
				"The controller with the lowest minimum-coord value must consume the one with the higher coords");
		}

		Set<IMultiblockPart> partsToAcquire = new HashSet<>(other.connectedParts);

		// releases all blocks and references gently, so they can be incorporated into another multiblock
		other._onAssimilated(this);

		for (IMultiblockPart acquiredPart : partsToAcquire) {
			// By definition, none of these can be the minimum block.
			if (acquiredPart.isInvalid()) {
				continue;
			}

			connectedParts.add(acquiredPart);
			acquiredPart.onAssimilated(this);
			this.onBlockAdded(acquiredPart);
		}

		this.onAssimilate(other);
		other.onAssimilated(this);
	}

	/**
	 * Called when this machine is consumed by another controller. Essentially,
	 * forcibly tear down this object.
	 *
	 * @param otherController {@link MultiblockControllerBase} The controller consuming this controller.
	 */
	private void _onAssimilated(MultiblockControllerBase otherController) {
		if (referenceCoord == null) { return; }

		if (WorldUtils.isChunkLoaded(worldObj, referenceCoord)) {
			class_2586 te = this.worldObj.method_8321(referenceCoord);
			if (te instanceof IMultiblockPart) {
				((IMultiblockPart) te).forfeitMultiblockSaveDelegate();
			}
		}
		this.referenceCoord = null;

		connectedParts.clear();
	}

	/**
	 * Callback. Called after this controller assimilates all the blocks from
	 * another controller. Use this to absorb that controller's game data.
	 *
	 * @param assimilated {@link MultiblockControllerBase} The controller whose uniqueness was added to our own.
	 */
	protected abstract void onAssimilate(MultiblockControllerBase assimilated);

	/**
	 * Callback. Called after this controller is assimilated into another
	 * controller. All blocks have been stripped out of this object and handed
	 * over to the other controller. This is intended primarily for cleanup.
	 *
	 * @param assimilator {@link MultiblockControllerBase} The controller which has assimilated this controller.
	 */
	protected abstract void onAssimilated(MultiblockControllerBase assimilator);

	/**
	 * Driver for the update loop. If the machine is assembled, runs the game
	 * logic update method.
	 */
	public final void updateMultiblockEntity() {
		if (connectedParts.isEmpty()) {
			// This shouldn't happen, but just in case...
			MultiblockRegistry.addDeadController(this.worldObj, this);
			return;
		}

		if (this.assemblyState != AssemblyState.Assembled) {
			// Not assembled - don't run game logic
			return;
		}

		if (worldObj.method_8608()) {
			updateClient();
		} else if (updateServer()) {
			// If this returns true, the server has changed its internal data.
			// If our chunks are loaded (they should be), we must mark our
			// chunks as dirty.
			if (minimumCoord != null && maximumCoord != null
				&& this.worldObj.method_22343(this.minimumCoord, this.maximumCoord)) {
				int minChunkX = minimumCoord.method_10263() >> 4;
				int minChunkZ = minimumCoord.method_10260() >> 4;
				int maxChunkX = maximumCoord.method_10263() >> 4;
				int maxChunkZ = maximumCoord.method_10260() >> 4;

				for (int x = minChunkX; x <= maxChunkX; x++) {
					for (int z = minChunkZ; z <= maxChunkZ; z++) {
						// Ensure that we save our data, even if our save
						// delegate is in has no TEs.
						class_2818 chunkToSave = this.worldObj.method_8497(x, z);
						chunkToSave.method_12044();
					}
				}
			}
		}
		// Else: Server, but no need to save data.
	}

	/**
	 * The server-side update loop! Use this similarly to a {@link class_2586}'s update
	 * loop. You do not need to call your superclass' {@code update()} if you're
	 * directly derived from {@link MultiblockControllerBase}. This is a callback. Note
	 * that this will only be called when the machine is assembled.
	 *
	 * @return {@code boolean} True if the multiblock should save data, i.e. its internal game
	 * state has changed. False otherwise.
	 */
	protected abstract boolean updateServer();

	/**
	 * Client-side update loop. Generally, this shouldn't do anything, but if
	 * you want to do some interpolation or something, do it here.
	 */
	protected abstract void updateClient();

	// Validation helpers

	/**
	 * The "frame" consists of the outer edges of the machine, plus the corners.
	 *
	 * @param world {@link class_1937} The object for the world in which this controller is
	 *              located.
	 * @param x     {@code int} X coordinate of the block being tested
	 * @param y     {@code int} Y coordinate of the block being tested
	 * @param z     {@code int} Z coordinate of the block being tested
	 * @throws MultiblockValidationException if the tested block is not allowed on the machine's frame
	 */
	protected void isBlockGoodForFrame(class_1937 world, int x, int y, int z) throws MultiblockValidationException {
		throw new MultiblockValidationException(
			String.format("%d, %d, %d - Block is not valid for use in the machine's interior", x, y, z));
	}

	/**
	 * The top consists of the top face, minus the edges.
	 *
	 * @param world {@link class_1937} The object for the world in which this controller is
	 *              located.
	 * @param x     {@code int} X coordinate of the block being tested
	 * @param y     {@code int} Y coordinate of the block being tested
	 * @param z     {@code int} Z coordinate of the block being tested
	 * @throws MultiblockValidationException if the tested block is not allowed on the machine's top face
	 */
	protected void isBlockGoodForTop(class_1937 world, int x, int y, int z) throws MultiblockValidationException {
		throw new MultiblockValidationException(
			String.format("%d, %d, %d - Block is not valid for use in the machine's interior", x, y, z));
	}

	/**
	 * The bottom consists of the bottom face, minus the edges.
	 *
	 * @param world {@link class_1937} The object for the world in which this controller is
	 *              located.
	 * @param x     {@code int} X coordinate of the block being tested
	 * @param y     {@code int} Y coordinate of the block being tested
	 * @param z     {@code int} Z coordinate of the block being tested
	 * @throws MultiblockValidationException if the tested block is not allowed on the machine's bottom
	 *                                       face
	 */
	protected void isBlockGoodForBottom(class_1937 world, int x, int y, int z) throws MultiblockValidationException {
		throw new MultiblockValidationException(
			String.format("%d, %d, %d - Block is not valid for use in the machine's interior", x, y, z));
	}

	/**
	 * The sides consist of the N/E/S/W-facing faces, minus the edges.
	 *
	 * @param world {@link class_1937} The object for the world in which this controller is
	 *              located.
	 * @param x     {@code int} X coordinate of the block being tested
	 * @param y     {@code int} Y coordinate of the block being tested
	 * @param z     {@code int} Z coordinate of the block being tested
	 * @throws MultiblockValidationException if the tested block is not allowed on the machine's side
	 *                                       faces
	 */
	protected void isBlockGoodForSides(class_1937 world, int x, int y, int z) throws MultiblockValidationException {
		throw new MultiblockValidationException(
			String.format("%d, %d, %d - Block is not valid for use in the machine's interior", x, y, z));
	}

	/**
	 * The interior is any block that does not touch blocks outside the machine.
	 *
	 * @param world {@link class_1937} The object for the world in which this controller is
	 *              located.
	 * @param x     {@code int} X coordinate of the block being tested
	 * @param y     {@code int} Y coordinate of the block being tested
	 * @param z     {@code int} Z coordinate of the block being tested
	 * @throws MultiblockValidationException if the tested block is not allowed in the machine's interior
	 */
	protected void isBlockGoodForInterior(class_1937 world, int x, int y, int z) throws MultiblockValidationException {
		throw new MultiblockValidationException(
			String.format("%d, %d, %d - Block is not valid for use in the machine's interior", x, y, z));
	}

	/**
	 * @return {@link class_2338} The reference coordinate, the block with the lowest x, y, z
	 * coordinates, evaluated in that order.
	 */
	public class_2338 getReferenceCoord() {
		if (referenceCoord == null) {
			selectNewReferenceCoord();
		}
		return referenceCoord;
	}

	/**
	 * @return {@code int} The number of blocks connected to this controller.
	 */
	public int getNumConnectedBlocks() {
		return connectedParts.size();
	}

	public abstract void write(class_2487 data);

	public abstract void read(class_2487 data);

	/**
	 * Force this multiblock to recalculate its minimum and maximum coordinates
	 * from the list of connected parts.
	 */
	public void recalculateMinMaxCoords() {
		int minX, minY, minZ;
		int maxX, maxY, maxZ;
		minX = minY = minZ = Integer.MAX_VALUE;
		maxX = maxY = maxZ = Integer.MIN_VALUE;

		for (IMultiblockPart part : connectedParts) {
			class_2338 pos = part.method_11016();
			if (pos.method_10263() < minX) {
				minX = pos.method_10263();
			}
			if (pos.method_10263() > maxX) {
				maxX = pos.method_10263();
			}
			if (pos.method_10264() < minY) {
				minY = pos.method_10264();
			}
			if (pos.method_10264() > maxY) {
				maxY = pos.method_10264();
			}
			if (pos.method_10260() < minZ) {
				minZ = pos.method_10260();
			}
			if (pos.method_10260() > maxZ) {
				maxZ = pos.method_10260();
			}
		}
		this.minimumCoord = new class_2338(minX, minY, minZ);
		this.maximumCoord = new class_2338(maxX, maxY, maxZ);
	}

	/**
	 * @return {@link class_2338} The minimum bounding-box coordinate containing this machine's
	 * blocks.
	 */
	public class_2338 getMinimumCoord() {
		if (minimumCoord == null) {
			recalculateMinMaxCoords();
		}
		return minimumCoord;
	}

	/**
	 * @return {@link class_2338} The maximum bounding-box coordinate containing this machine's
	 * blocks.
	 */
	public class_2338 getMaximumCoord() {
		if (maximumCoord == null) {
			recalculateMinMaxCoords();
		}
		return maximumCoord;
	}

	/**
	 * Called when the save delegate's {@link class_2586} entity is being asked for its
	 * description packet
	 *
	 * @param data {@link class_2487} A fresh compound tag to write your multiblock data into
	 */
	public abstract void formatDescriptionPacket(class_2487 data);

	/**
	 * Called when the save delegate's {@link class_2586} entity receiving a description
	 * packet
	 *
	 * @param data {@link class_2487} A compound tag containing multiblock data to import
	 */
	public abstract void decodeDescriptionPacket(class_2487 data);

	/**
	 * @return {@code boolean} True if this controller has no associated blocks, false otherwise
	 */
	public boolean isEmpty() {
		return connectedParts.isEmpty();
	}

	/**
	 * Tests whether this multiblock should consume the other multiblock and
	 * become the new multiblock master when the two multiblocks are adjacent.
	 * Assumes both multiblocks are the same type.
	 *
	 * @param otherController {@link MultiblockControllerBase} The other multiblock controller.
	 * @return {@code boolean} True if this multiblock should consume the other, false
	 * otherwise.
	 * @throws IllegalArgumentException The two multiblocks have different master classes
	 * @throws IllegalArgumentException The two controllers with the same reference coord both
	 *                                  have valid parts
	 */
	public boolean shouldConsume(MultiblockControllerBase otherController) {
		if (!otherController.getClass().equals(getClass())) {
			throw new IllegalArgumentException(
				"Attempting to merge two multiblocks with different master classes - this should never happen!");
		}

		if (otherController == this) {
			return false;
		} // Don't be silly, don't eat yourself.

		int res = _shouldConsume(otherController);
		if (res < 0) {
			return true;
		} else if (res > 0) {
			return false;
		} else {
			// Strip dead parts from both and retry
			RebornCore.LOGGER.warn(
				String.format("[%s] Encountered two controllers with the same reference coordinate. Auditing connected parts and retrying.",
					worldObj.method_8608() ? "CLIENT" : "SERVER"));
			auditParts();
			otherController.auditParts();

			res = _shouldConsume(otherController);
			if (res < 0) {
				return true;
			} else if (res > 0) {
				return false;
			} else {
				RebornCore.LOGGER.error(String.format("My Controller (%d): size (%d), parts: %s", hashCode(), connectedParts.size(),
					getPartsListString()));
				RebornCore.LOGGER.error(String.format("Other Controller (%d): size (%d), coords: %s", otherController.hashCode(),
					otherController.connectedParts.size(), otherController.getPartsListString()));
				throw new IllegalArgumentException("[" + (worldObj.method_8608() ? "CLIENT" : "SERVER")
					+ "] Two controllers with the same reference coord that somehow both have valid parts - this should never happen!");
			}

		}
	}

	private int _shouldConsume(MultiblockControllerBase otherController) {
		class_2338 myCoord = getReferenceCoord();
		class_2338 theirCoord = otherController.getReferenceCoord();

		// Always consume other controllers if their reference coordinate is
		// null - this means they're empty and can be assimilated on the cheap
		if (theirCoord == null) {
			return -1;
		} else {
			return myCoord.method_10265(theirCoord);
		}
	}

	private String getPartsListString() {
		StringBuilder sb = new StringBuilder();
		boolean first = true;
		for (IMultiblockPart part : connectedParts) {
			if (!first) {
				sb.append(", ");
			}
			sb.append(String.format("(%d: %d, %d, %d)", part.hashCode(), part.method_11016().method_10263(), part.method_11016().method_10264(),
				part.method_11016().method_10260()));
			first = false;
		}

		return sb.toString();
	}

	/**
	 * Checks all the parts in the controller. If any are dead or do not
	 * exist in the world, they are removed.
	 */
	private void auditParts() {
		HashSet<IMultiblockPart> deadParts = new HashSet<>();
		for (IMultiblockPart part : connectedParts) {
			if (part.isInvalid() || worldObj.method_8321(part.method_11016()) != part) {
				onDetachBlock(part);
				deadParts.add(part);
			}
		}

		connectedParts.removeAll(deadParts);
		RebornCore.LOGGER.warn(String.format("[%s] Controller found %d dead parts during an audit, %d parts remain attached",
			worldObj.method_8608() ? "CLIENT" : "SERVER", deadParts.size(), connectedParts.size()));
	}

	/**
	 * Called when this machine may need to check for blocks that are no longer
	 * physically connected to the reference coordinate.
	 *
	 * @return {@link Set} Set with removed {@link IMultiblockPart}s.
	 */
	public Set<IMultiblockPart> checkForDisconnections() {
		if (!this.shouldCheckForDisconnections) {
			return null;
		}

		if (this.isEmpty()) {
			MultiblockRegistry.addDeadController(worldObj, this);
			return null;
		}

		// Invalidate our reference coord, we'll recalculate it shortly
		referenceCoord = null;

		// Reset visitations and find the minimum coordinate
		Set<IMultiblockPart> deadParts = new HashSet<>();
		class_2338 pos;
		IMultiblockPart referencePart = null;

		int originalSize = connectedParts.size();

		for (IMultiblockPart part : connectedParts) {
			pos = part.getWorldLocation();
			if (!WorldUtils.isChunkLoaded(worldObj, pos) || part.isInvalid()) {
				deadParts.add(part);
				onDetachBlock(part);
				continue;
			}

			if (worldObj.method_8321(pos) != part) {
				deadParts.add(part);
				onDetachBlock(part);
				continue;
			}

			part.setUnvisited();
			part.forfeitMultiblockSaveDelegate();

			if (referenceCoord == null) {
				referenceCoord = pos;
				referencePart = part;
			} else if (pos.method_10265(referenceCoord) < 0) {
				referenceCoord = pos;
				referencePart = part;
			}
		}

		connectedParts.removeAll(deadParts);
		deadParts.clear();

		if (referencePart == null || isEmpty()) {
			// There are no valid parts remaining. The entire multiblock was
			// unloaded during a chunk unload. Halt.
			shouldCheckForDisconnections = false;
			MultiblockRegistry.addDeadController(worldObj, this);
			return null;
		} else {
			referencePart.becomeMultiblockSaveDelegate();
		}

		// Now visit all connected parts, breadth-first, starting from reference
		// coord's part
		IMultiblockPart part;
		LinkedList<IMultiblockPart> partsToCheck = new LinkedList<>();
		IMultiblockPart[] nearbyParts;
		int visitedParts = 0;

		partsToCheck.add(referencePart);

		while (!partsToCheck.isEmpty()) {
			part = partsToCheck.removeFirst();
			part.setVisited();
			visitedParts++;

			// Chunk-safe on server, but not on client
			nearbyParts = part.getNeighboringParts();
			for (IMultiblockPart nearbyPart : nearbyParts) {
				// Ignore different machines
				if (nearbyPart.getMultiblockController() != this) {
					continue;
				}

				if (!nearbyPart.isVisited()) {
					nearbyPart.setVisited();
					partsToCheck.add(nearbyPart);
				}
			}
		}

		// Finally, remove all parts that remain disconnected.
		Set<IMultiblockPart> removedParts = new HashSet<>();
		for (IMultiblockPart orphanCandidate : connectedParts) {
			if (!orphanCandidate.isVisited()) {
				deadParts.add(orphanCandidate);
				orphanCandidate.onOrphaned(this, originalSize, visitedParts);
				onDetachBlock(orphanCandidate);
				removedParts.add(orphanCandidate);
			}
		}

		// Trim any blocks that were invalid, or were removed.
		connectedParts.removeAll(deadParts);

		// Cleanup. Not necessary, really.
		deadParts.clear();

		// Juuuust in case.
		if (referenceCoord == null) {
			selectNewReferenceCoord();
		}

		// We've run the checks from here on out.
		shouldCheckForDisconnections = false;

		return removedParts;
	}

	/**
	 * Detach all parts. Return a set of all parts which still have a valid {@link class_2586}
	 * entity. Chunk-safe.
	 *
	 * @return {@link Set} A set of all {@link IMultiblockPart}s which still have a valid {@link class_2586} entity.
	 */
	public Set<IMultiblockPart> detachAllBlocks() {
		if (worldObj == null) {
			return new HashSet<>();
		}

		for (IMultiblockPart part : connectedParts) {
			if (WorldUtils.isChunkLoaded(worldObj, part.getWorldLocation())) {
				onDetachBlock(part);
			}
		}

		Set<IMultiblockPart> detachedParts = connectedParts;
		connectedParts = new HashSet<>();
		return detachedParts;
	}

	/**
	 * @return {@code boolean} True if this multiblock machine is considered assembled and ready
	 * to go.
	 */
	public boolean isAssembled() {
		return this.assemblyState == AssemblyState.Assembled;
	}

	private void selectNewReferenceCoord() {
		IMultiblockPart theChosenOne = null;
		class_2338 pos;
		referenceCoord = null;

		for (IMultiblockPart part : connectedParts) {
			pos = part.getWorldLocation();
			if (part.isInvalid() || !WorldUtils.isChunkLoaded(worldObj, pos)) {
				// Chunk is unloading, skip this coord to prevent chunk thrashing
				continue;
			}

			if (referenceCoord == null || referenceCoord.method_10265(pos) > 0) {
				referenceCoord = pos;
				theChosenOne = part;
			}
		}

		if (theChosenOne != null) {
			theChosenOne.becomeMultiblockSaveDelegate();
		}
	}

	/**
	 * Marks the reference coord dirty & updateable.
	 * <p>
	 *  On the server, this will mark the for a data-update, so that nearby
	 *  clients will receive an updated description packet from the server after
	 *  a short time. The block's chunk will also be marked dirty and the block's
	 *  chunk will be saved to disk the next time chunks are saved.
	 * </p>
	 * <p>On the client, this will mark the block for a rendering update.</p>
	 */
	protected void markReferenceCoordForUpdate() {
		class_2338 rc = getReferenceCoord();
		if (worldObj != null && rc != null) {
			//TO-DO Change to notifyBlockUpdate, probably
			WorldUtils.updateBlock(worldObj, rc);
		}
	}

	/**
	 * Marks the reference coord dirty.
	 * <p>
	 *  On the server, this marks the reference coord's chunk as dirty; the block
	 *  (and chunk) will be saved to disk the next time chunks are saved. This
	 *  does NOT mark it dirty for a description-packet update.
	 * </p>
	 * <p>On the client, does nothing.</p>
	 *
	 * @see MultiblockControllerBase#markReferenceCoordForUpdate()
	 */
	protected void markReferenceCoordDirty() {
		if (worldObj == null || worldObj.method_8608()) {
			return;
		}

		class_2338 referenceCoord = getReferenceCoord();
		if (referenceCoord == null) {
			return;
		}

		worldObj.method_8524(referenceCoord);
	}

}
