/*
 * This file is part of RebornCore, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2021 TeamReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.chunkloading;

import com.mojang.serialization.Codec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import io.netty.buffer.ByteBuf;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import reborncore.common.network.NetworkManager;
import reborncore.common.network.clientbound.ChunkSyncPayload;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import net.minecraft.class_10741;
import net.minecraft.class_18;
import net.minecraft.class_1923;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2960;
import net.minecraft.class_3215;
import net.minecraft.class_3218;
import net.minecraft.class_3222;
import net.minecraft.class_3230;
import net.minecraft.class_5321;
import net.minecraft.class_9135;
import net.minecraft.class_9139;

// This does not do the actual chunk loading, just keeps track of what chunks the chunk loader has loaded
public class ChunkLoaderManager extends class_18 {

	public static Codec<ChunkLoaderManager> CODEC = Codec.list(LoadedChunk.CODEC).xmap(ChunkLoaderManager::fromChunks, ChunkLoaderManager::getLoadedChunks);
	public static final class_10741<ChunkLoaderManager> TYPE = new class_10741<>("chunk_loader", ChunkLoaderManager::new, CODEC, null);

	private static class_3230 CHUNK_LOADER;
	private static final String KEY = "reborncore_chunk_loader";
	private static final int RADIUS = 1;

	public static void register() {
		CHUNK_LOADER = class_3230.method_66026("reborncore:chunk_loader", 0L, 15);
	}

	public ChunkLoaderManager() {
	}

	public static ChunkLoaderManager get(class_1937 world) {
		class_3218 serverWorld = (class_3218) world;
		return serverWorld.method_17983().method_17924(TYPE);
	}

	private final List<LoadedChunk> loadedChunks = new ArrayList<>();

	public static ChunkLoaderManager fromChunks(List<LoadedChunk> chunks) {
		ChunkLoaderManager chunkLoaderManager = new ChunkLoaderManager();

		chunkLoaderManager.loadedChunks.clear();

		chunkLoaderManager.loadedChunks.addAll(chunks);

		return chunkLoaderManager;
	}

	public List<LoadedChunk> getLoadedChunks() {
		return loadedChunks;
	}

	public Optional<LoadedChunk> getLoadedChunk(class_1937 world, class_1923 chunkPos, class_2338 chunkLoader){
		return loadedChunks.stream()
			.filter(loadedChunk -> loadedChunk.world().equals(getWorldName(world)))
			.filter(loadedChunk -> loadedChunk.chunk().equals(chunkPos))
			.filter(loadedChunk -> loadedChunk.chunkLoader().equals(chunkLoader))
			.findFirst();
	}

	public Optional<LoadedChunk> getLoadedChunk(class_1937 world, class_1923 chunkPos){
		return loadedChunks.stream()
			.filter(loadedChunk -> loadedChunk.world().equals(getWorldName(world)))
			.filter(loadedChunk -> loadedChunk.chunk().equals(chunkPos))
			.findFirst();
	}

	public List<LoadedChunk> getLoadedChunks(class_1937 world, class_2338 chunkLoader){
		return loadedChunks.stream()
			.filter(loadedChunk -> loadedChunk.world().equals(getWorldName(world)))
			.filter(loadedChunk -> loadedChunk.chunkLoader().equals(chunkLoader))
			.collect(Collectors.toList());
	}

	public boolean isChunkLoaded(class_1937 world, class_1923 chunkPos, class_2338 chunkLoader){
		return getLoadedChunk(world, chunkPos, chunkLoader).isPresent();
	}

	public boolean isChunkLoaded(class_1937 world, class_1923 chunkPos){
		return getLoadedChunk(world, chunkPos).isPresent();
	}


	public void loadChunk(class_1937 world, class_1923 chunkPos, class_2338 chunkLoader, String player){
		Validate.isTrue(!isChunkLoaded(world, chunkPos, chunkLoader), "chunk is already loaded");
		LoadedChunk loadedChunk = new LoadedChunk(chunkPos, getWorldName(world), player, chunkLoader);
		loadedChunks.add(loadedChunk);

		loadChunk((class_3218) world, loadedChunk);

		method_80();
	}

	public void unloadChunkLoader(class_1937 world, class_2338 chunkLoader){
		getLoadedChunks(world, chunkLoader).forEach(loadedChunk -> unloadChunk(world, loadedChunk.chunk(), chunkLoader));
	}

	public void unloadChunk(class_1937 world, class_1923 chunkPos, class_2338 chunkLoader){
		Optional<LoadedChunk> optionalLoadedChunk = getLoadedChunk(world, chunkPos, chunkLoader);
		Validate.isTrue(optionalLoadedChunk.isPresent(), "chunk is not loaded");

		LoadedChunk loadedChunk = optionalLoadedChunk.get();

		loadedChunks.remove(loadedChunk);

		if(!isChunkLoaded(world, loadedChunk.chunk())){
			final class_3215 serverChunkManager = ((class_3218) world).method_14178();
			serverChunkManager.method_66010(ChunkLoaderManager.CHUNK_LOADER, loadedChunk.chunk(), RADIUS);
		}
		method_80();
	}

	public void onServerWorldLoad(class_3218 world) {
		loadedChunks.forEach(loadedChunk -> loadChunk(world, loadedChunk));
	}

	public void onServerWorldTick(class_3218 world) {
		if (!loadedChunks.isEmpty()) {
			world.method_14197();
		}
	}

	public static class_2960 getWorldName(class_1937 world){
		return world.method_27983().method_29177();
	}

	public static class_5321<class_1937> getDimensionRegistryKey(class_1937 world){
		return world.method_27983();
	}

	public void syncChunkLoaderToClient(class_3222 serverPlayerEntity, class_2338 chunkLoader){
		syncToClient(serverPlayerEntity, loadedChunks.stream().filter(loadedChunk -> loadedChunk.chunkLoader().equals(chunkLoader)).collect(Collectors.toList()));
	}

	public void syncAllToClient(class_3222 serverPlayerEntity) {
		syncToClient(serverPlayerEntity, loadedChunks);
	}

	public void clearClient(class_3222 serverPlayerEntity) {
		syncToClient(serverPlayerEntity, Collections.emptyList());
	}

	public void syncToClient(class_3222 serverPlayerEntity, List<LoadedChunk> chunks) {
		NetworkManager.sendToPlayer(new ChunkSyncPayload(chunks), serverPlayerEntity);
	}

	private void loadChunk(class_3218 world, LoadedChunk loadedChunk) {
		class_1923 chunkPos = loadedChunk.chunk();
		world.method_14178().method_66009(ChunkLoaderManager.CHUNK_LOADER, chunkPos, RADIUS);
	}

	public record LoadedChunk(class_1923 chunk, class_2960 world, String player, class_2338 chunkLoader) {
		public static Codec<class_1923> CHUNK_POS_CODEC = RecordCodecBuilder.create(instance ->
			instance.group(
					Codec.INT.fieldOf("x").forGetter(p -> p.field_9181),
					Codec.INT.fieldOf("z").forGetter(p -> p.field_9180)
				)
				.apply(instance, class_1923::new));

		public static Codec<LoadedChunk> CODEC = RecordCodecBuilder.create(instance ->
			instance.group(
					CHUNK_POS_CODEC.fieldOf("chunk").forGetter(LoadedChunk::chunk),
					class_2960.field_25139.fieldOf("world").forGetter(LoadedChunk::world),
					Codec.STRING.fieldOf("player").forGetter(LoadedChunk::player),
					class_2338.field_25064.fieldOf("chunkLoader").forGetter(LoadedChunk::chunkLoader)
				)
				.apply(instance, LoadedChunk::new));

		public static class_9139<ByteBuf, class_1923> CHUNK_POS_PACKET_CODEC = class_9139.method_56435(
			class_9135.field_49675, chunkPos -> chunkPos.field_9181,
			class_9135.field_49675, chunkPos -> chunkPos.field_9180,
			class_1923::new
		);

		public static class_9139<ByteBuf, LoadedChunk> PACKET_CODEC = class_9139.method_56905(
			CHUNK_POS_PACKET_CODEC, LoadedChunk::chunk,
			class_2960.field_48267, LoadedChunk::world,
			class_9135.field_48554, LoadedChunk::player,
			class_2338.field_48404, LoadedChunk::chunkLoader,
			LoadedChunk::new
		);

		public LoadedChunk {
			Validate.isTrue(!StringUtils.isBlank(player), "Player cannot be blank");
		}
	}
}
