/*
 * This file is part of RebornCore, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2021 TeamReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.blockentity;

import com.mojang.brigadier.exceptions.CommandSyntaxException;
import io.netty.buffer.ByteBuf;
import net.fabricmc.fabric.api.transfer.v1.item.InventoryStorage;
import net.fabricmc.fabric.api.transfer.v1.item.ItemStorage;
import net.fabricmc.fabric.api.transfer.v1.storage.StorageUtil;
import net.minecraft.class_11352;
import net.minecraft.class_11362;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_1263;
import net.minecraft.class_1799;
import net.minecraft.class_2350;
import net.minecraft.class_2487;
import net.minecraft.class_2522;
import net.minecraft.class_7225;
import net.minecraft.class_8942;
import net.minecraft.class_9135;
import net.minecraft.class_9139;
import org.apache.commons.lang3.Validate;
import org.jetbrains.annotations.Nullable;
import reborncore.common.util.NBTSerializable;
import reborncore.common.util.RebornInventory;

import java.util.*;
import java.util.stream.Collectors;

import static reborncore.RebornCore.LOGGER;

public class SlotConfiguration implements NBTSerializable {
	public static final class_9139<ByteBuf, SlotConfiguration> PACKET_CODEC = SlotConfigHolder.PACKET_CODEC
		.method_56433(class_9135.method_56363())
		.method_56432(SlotConfiguration::new, SlotConfiguration::getSlotDetails);

	List<SlotConfigHolder> slotDetails = new ArrayList<>();

	@Nullable
	class_1263 inventory;

	public SlotConfiguration(RebornInventory<?> inventory) {
		this.inventory = inventory;

		for (int i = 0; i < inventory.method_5439(); i++) {
			updateSlotDetails(new SlotConfigHolder(i));
		}
	}

	private SlotConfiguration(List<SlotConfigHolder> slotDetails) {
		this.slotDetails = slotDetails;
		this.inventory = null;
	}

	public void update(MachineBaseBlockEntity machineBase) {
		if (inventory == null && machineBase.getOptionalInventory().isPresent()) {
			inventory = machineBase.getOptionalInventory().get();
		}
		if (inventory != null && slotDetails.size() != inventory.method_5439()) {
			for (int i = 0; i < inventory.method_5439(); i++) {
				SlotConfigHolder holder = getSlotDetails(i);
				if (holder == null) {
					LOGGER.debug("Fixed slot " + i + " in " + machineBase);
					// hmm, something has gone wrong
					updateSlotDetails(new SlotConfigHolder(i));
				}
			}
		}
		if (!machineBase.method_10997().method_8608() && machineBase.method_10997().method_8510() % machineBase.slotTransferSpeed() == 0) {
			getSlotDetails().forEach(slotConfigHolder -> slotConfigHolder.handleItemIO(machineBase));
		}
	}

	public SlotConfiguration(class_11368 view) {
		read(view);
	}

	public List<SlotConfigHolder> getSlotDetails() {
		return slotDetails;
	}

	/**
	 * Replaces or adds a slot detail for the slot id
	 *
	 * @param slotConfigHolder {@link SlotConfigHolder}
	 * @return {@link SlotConfigHolder} Updated SlotConfigHolder
	 */
	public SlotConfigHolder updateSlotDetails(SlotConfigHolder slotConfigHolder) {
		SlotConfigHolder lookup = getSlotDetails(slotConfigHolder.slotID);
		if (lookup != null) {
			slotDetails.remove(lookup);
		}
		slotDetails.add(slotConfigHolder);
		return slotConfigHolder;
	}

	@Nullable
	public SlotConfigHolder getSlotDetails(int id) {
		for (SlotConfigHolder detail : slotDetails) {
			if (detail.slotID == id) {
				return detail;
			}
		}
		return null;
	}

	public List<SlotConfig> getSlotsForSide(class_2350 facing) {
		return slotDetails.stream().map(slotConfigHolder -> slotConfigHolder.getSideDetail(facing)).collect(Collectors.toList());
	}

	@Override
	public void write(class_11372 view) {
		view.method_71465("size", slotDetails.size());
		for (int i = 0; i < slotDetails.size(); i++) {
			slotDetails.get(i).write(view.method_71461("slot_" + i));
		}
	}

	@Override
	public void read(class_11368 view) {
		int size = view.method_71424("size", 0);
		for (int i = 0; i < size; i++) {
			view.method_71420("slot_" + i).ifPresent(slot -> {
				updateSlotDetails(new SlotConfigHolder(slot));
			});
		}
	}

	public static class SlotConfigHolder implements NBTSerializable {
		public static final class_9139<ByteBuf, SlotConfigHolder> PACKET_CODEC = class_9139.method_58025(
			class_9135.field_49675, SlotConfigHolder::getSlotID,
			class_9135.method_56377(HashMap::new, class_2350.field_48450, SlotConfig.PACKET_CODEC), SlotConfigHolder::getSideMap,
			class_9135.field_48547, SlotConfigHolder::autoInput,
			class_9135.field_48547, SlotConfigHolder::autoOutput,
			class_9135.field_48547, SlotConfigHolder::filter,
			class_9135.field_49675, SlotConfigHolder::getPriority,
			SlotConfigHolder::new
		);

		int slotID;
		HashMap<class_2350, SlotConfig> sideMap;
		boolean input, output, filter;
		@Nullable
		public class_2350 first, last;

		private SlotConfigHolder(int slotID, HashMap<class_2350, SlotConfig> sideMap, boolean input, boolean output, boolean filter, int priority) {
			this.slotID = slotID;
			this.sideMap = sideMap;
			this.input = input;
			this.output = output;
			this.filter = filter;
			setPriority(priority);
		}

		public SlotConfigHolder(int slotID) {
			this.slotID = slotID;
			sideMap = new HashMap<>();
			Arrays.stream(class_2350.values()).forEach(facing -> sideMap.put(facing, new SlotConfig(facing, slotID)));
		}

		public SlotConfigHolder(class_11368 view) {
			sideMap = new HashMap<>();
			read(view);
			Validate.isTrue(Arrays.stream(class_2350.values())
								.map(enumFacing -> sideMap.get(enumFacing))
								.noneMatch(Objects::isNull),
							"sideMap failed to load from nbt"
			);
		}

		public SlotConfig getSideDetail(class_2350 side) {
			Validate.notNull(side, "A none null side must be used");
			SlotConfig slotConfig = sideMap.get(side);
			Validate.notNull(slotConfig, "slotConfig was null for side " + side);
			return slotConfig;
		}

		public void updateSlotConfig(SlotConfig config) {
			SlotConfig toEdit = sideMap.get(config.side);
			toEdit.slotIO = config.slotIO;
		}

		private void handleItemIO(MachineBaseBlockEntity machineBase) {
			if (!input && !output) {
				return;
			}
			if (first != null) {
				handleItemSideIo(machineBase, sideMap.get(first));
			}
			sideMap.forEach((key, config) -> {
				if (key == first || key == last) return;
				handleItemSideIo(machineBase, config);
			});
			if (last != null) {
				handleItemSideIo(machineBase, sideMap.get(last));
			}
		}

		private void handleItemSideIo(MachineBaseBlockEntity machineBase, SlotConfig config) {
			switch (config.getSlotIO().getIoConfig()) {
				case INPUT -> {
					if (input) config.handleItemInput(machineBase);
				}
				case OUTPUT -> {
					if (output) config.handleItemOutput(machineBase);
				}
			}
		}

		public boolean autoInput() {
			return input;
		}

		public boolean autoOutput() {
			return output;
		}

		public boolean filter() {
			return filter;
		}

		public int getPriority() {
			int first = this.first == null ? 6 : this.first.ordinal();
			int last = this.last == null ? 6 : this.last.ordinal();
			return first * 10 + last;
		}

		public void setPriority(int priority) {
			int first = priority / 10;
			int last = priority % 10;
			class_2350[] directions = class_2350.values();
			this.first = first == 6 ? null : directions[first];
			this.last = last == 6 ? null : directions[last];
		}

		public void setInput(boolean input) {
			this.input = input;
		}

		public void setOutput(boolean output) {
			this.output = output;
		}

		public void setFilter(boolean filter) {
			this.filter = filter;
		}

		public int getSlotID() {
			return slotID;
		}

		public HashMap<class_2350, SlotConfig> getSideMap() {
			return sideMap;
		}

		@Override
		public void write(class_11372 view) {
			view.method_71465("slotID", slotID);
			Arrays.stream(class_2350.values()).forEach(facing -> sideMap.get(facing).write(view.method_71461("side_" + facing.ordinal())));
			view.method_71472("input", input);
			view.method_71472("output", output);
			view.method_71472("filter", filter);
			if (this.first != null || this.last != null) {
				view.method_71465("priority", getPriority());
			}
		}

		@Override
		public void read(class_11368 view) {
			sideMap.clear();
			slotID = view.method_71424("slotID", 0);
			Arrays.stream(class_2350.values()).forEach(facing -> {
				view.method_71420("side_" + facing.ordinal()).ifPresent(config -> {
					sideMap.put(facing, new SlotConfig(config));
				});
			});
			input = view.method_71433("input", false);
			output = view.method_71433("output", false);
			filter = view.method_71433("filter", false);
			view.method_71439("priority").ifPresentOrElse(this::setPriority, () -> {
				first = null;
				last = null;
			});
		}
	}

	public static class SlotConfig implements NBTSerializable {
		public static final class_9139<ByteBuf, SlotConfig> PACKET_CODEC = class_9139.method_56436(
			class_2350.field_48450, SlotConfig::getSide,
			SlotIO.PACKET_CODEC, SlotConfig::getSlotIO,
			class_9135.field_49675, SlotConfig::getSlotID,
			SlotConfig::new
		);

		private class_2350 side;
		private SlotIO slotIO;
		private int slotID;

		public SlotConfig(class_2350 side, int slotID) {
			this.side = side;
			this.slotID = slotID;
			this.slotIO = new SlotIO(ExtractConfig.NONE);
		}

		public SlotConfig(class_2350 side, SlotIO slotIO, int slotID) {
			this.side = side;
			this.slotIO = slotIO;
			this.slotID = slotID;
		}

		public SlotConfig(class_11368 view) {
			read(view);
			Validate.notNull(side, "error when loading slot config");
			Validate.notNull(slotIO, "error when loading slot config");
		}

		public class_2350 getSide() {
			Validate.notNull(side, "side is null");
			return side;
		}

		public SlotIO getSlotIO() {
			Validate.notNull(slotIO, "error when loading slot config");
			return slotIO;
		}

		public int getSlotID() {
			return slotID;
		}

		private void handleItemInput(MachineBaseBlockEntity machineBase) {
			RebornInventory<?> inventory = machineBase.getOptionalInventory().get();
			class_1799 targetStack = inventory.method_5438(slotID);
			if (targetStack.method_7914() == targetStack.method_7947()) {
				return;
			}

			StorageUtil.move(
					ItemStorage.SIDED.find(machineBase.method_10997(), machineBase.method_11016().method_10093(side), side.method_10153()),
					InventoryStorage.of(machineBase, null).getSlot(slotID),
					iv -> true,
					4, // Move up to 4 per tick.
					null
			);
		}

		private void handleItemOutput(MachineBaseBlockEntity machineBase) {
			RebornInventory<?> inventory = machineBase.getOptionalInventory().get();
			class_1799 sourceStack = inventory.method_5438(slotID);
			if (sourceStack.method_7960()) {
				return;
			}

			StorageUtil.move(
					InventoryStorage.of(machineBase, null).getSlot(slotID),
					ItemStorage.SIDED.find(machineBase.method_10997(), machineBase.method_11016().method_10093(side), side.method_10153()),
					iv -> true,
					Long.MAX_VALUE,
					null
			);
		}

		@Override
		public void write(class_11372 view) {
			view.method_71465("side", side.ordinal());
			slotIO.write(view.method_71461("config"));
			view.method_71465("slot", slotID);
		}

		@Override
		public void read(class_11368 view) {
			side = class_2350.values()[view.method_71424("side", 0)];
			view.method_71420("config").ifPresent(config -> {
				slotIO = new SlotIO(config);
			});
			slotID = view.method_71424("slot", 0);
		}
	}

	public static class SlotIO implements NBTSerializable {
		public static final class_9139<ByteBuf, SlotIO> PACKET_CODEC = class_9139.method_56434(
			ExtractConfig.PACKET_CODEC, SlotIO::getIoConfig,
			SlotIO::new
		);

		ExtractConfig ioConfig;

		public SlotIO(class_11368 view) {
			read(view);
		}

		public SlotIO(ExtractConfig ioConfig) {
			this.ioConfig = ioConfig;
		}

		public ExtractConfig getIoConfig() {
			return ioConfig;
		}

		@Override
		public void write(class_11372 view) {
			view.method_71465("config", ioConfig.ordinal());
		}

		@Override
		public void read(class_11368 view) {
			ioConfig = ExtractConfig.values()[view.method_71424("config", 0)];
		}
	}

	public enum ExtractConfig {
		NONE(false, false),
		INPUT(false, true),
		OUTPUT(true, false);

		public static final class_9139<ByteBuf, ExtractConfig> PACKET_CODEC = class_9135.field_49675
			.method_56432(integer -> ExtractConfig.values()[integer], Enum::ordinal);

		boolean extract;
		boolean insert;

		ExtractConfig(boolean extract, boolean insert) {
			this.extract = extract;
			this.insert = insert;
		}

		public boolean isExtract() {
			return extract;
		}

		public boolean isInsert() {
			return insert;
		}

		public ExtractConfig getNext() {
			int i = this.ordinal() + 1;
			if (i >= ExtractConfig.values().length) {
				i = 0;
			}
			return ExtractConfig.values()[i];
		}
	}

	public String toJson(String machineIdent, class_7225.class_7874 registryLookup) {
		try (class_8942.class_11340 logging = new class_8942.class_11340(() -> "SlotConfiguration", LOGGER)) {
			class_11362 view = class_11362.method_71459(logging, registryLookup);
			write(view.method_71461("data"));
			view.method_71469("machine", machineIdent);
			return view.method_71475().toString();
		}
	}

	public void readJson(String json, String machineIdent, class_7225.class_7874 registryLookup) throws UnsupportedOperationException {
		class_2487 compound;
		try {
			compound = class_2522.method_67315(json);
		} catch (CommandSyntaxException e) {
			throw new UnsupportedOperationException("Clipboard contents isn't a valid slot configuration");
		}
		if (!compound.method_10545("machine") || !compound.method_10558("machine").orElseThrow().equals(machineIdent)) {
			throw new UnsupportedOperationException("Machine config is not for this machine.");
		}
		try (class_8942.class_11340 logging = new class_8942.class_11340(() -> "SlotConfiguration", LOGGER)) {
			read(class_11352.method_71417(logging, registryLookup, compound.method_68568("data")));
		}
	}

	// DO NOT CALL THIS, use the inventory access on the inventory
	public static boolean canInsertItem(int index, class_1799 itemStackIn, class_2350 direction, MachineBaseBlockEntity blockEntity) {
		if(itemStackIn.method_7960()){
			return false;
		}
		SlotConfiguration.SlotConfigHolder slotConfigHolder = blockEntity.getSlotConfiguration().getSlotDetails(index);
		SlotConfiguration.SlotConfig slotConfig = slotConfigHolder.getSideDetail(direction);
		if (slotConfig.getSlotIO().getIoConfig().isInsert()) {
			if (slotConfigHolder.filter()) {
				if(blockEntity instanceof SlotFilter){
					return ((SlotFilter) blockEntity).isStackValid(index, itemStackIn);
				}
			}
			return blockEntity.method_5437(index, itemStackIn);
		}
		return false;
	}

	// DO NOT CALL THIS, use the inventory access on the inventory
	public static boolean canExtractItem(int index, class_1799 stack, class_2350 direction, MachineBaseBlockEntity blockEntity) {
		SlotConfiguration.SlotConfigHolder slotConfigHolder = blockEntity.getSlotConfiguration().getSlotDetails(index);
		SlotConfiguration.SlotConfig slotConfig = slotConfigHolder.getSideDetail(direction);
		if (slotConfig.getSlotIO().getIoConfig().isExtract()) {
			return true;
		}
		return false;
	}

	public interface SlotFilter {
		boolean isStackValid(int slotID, class_1799 stack);

		int[] getInputSlots();
	}

}
