/*
 * This file is part of RebornCore, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2021 TeamReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.blockentity;

import java.util.HashSet;
import java.util.Set;
import java.util.function.BiPredicate;
import net.minecraft.class_1922;
import net.minecraft.class_2246;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2680;

/**
 * Writes a multiblock for either verification or hologram rendering
 *
 * @author ramidzkh
 * @see MultiblockVerifier
 */
public interface MultiblockWriter {

	/**
	 * Adds a block to the multiblock
	 *
	 * @param x         {@code int} X
	 * @param y         {@code int} Y
	 * @param z         {@code int} Z
	 * @param predicate {@link BiPredicate} Predicate of the position
	 * @param state     {@link class_2680} The state for the hologram
	 * @return {@link MultiblockWriter} This. Useful for chaining
	 */
	MultiblockWriter add(int x, int y, int z, BiPredicate<class_1922, class_2338> predicate, class_2680 state);

	/**
	 * Fills a section between (ax, ay, az) to (bx, by, bz)
	 *
	 * @param ax        {@code int} X of the first point
	 * @param ay        {@code int} Y of the first point
	 * @param az        {@code int} Z of the first point
	 * @param bx        {@code int} X of the second point
	 * @param by        {@code int} X of the second point
	 * @param bz        {@code int} Z of the second point
	 * @param predicate {@link BiPredicate} Predicate of the position
	 * @param state     {@link class_2680} The state for the hologram
	 * @return {@link MultiblockWriter} This. Useful for chaining
	 */
	default MultiblockWriter fill(int ax, int ay, int az, int bx, int by, int bz, BiPredicate<class_1922, class_2338> predicate, class_2680 state) {
		for (int x = ax; x < bx; x++) {
			for (int y = ay; y < by; y++) {
				for (int z = az; z < bz; z++) {
					add(x, y, z, predicate, state);
				}
			}
		}

		return this;
	}

	/**
	 * Fills the outer ring of (0, 0, 0) to (pX, pY, pZ) through the axis, using the <code>predicate</code> and
	 * <code>state</code>. The inside of the ring uses <code>holePredicate</code> and <code>holeHologramState</code>
	 *
	 * @param through           {@link class_2350.class_2351} The axis to go through
	 * @param pX                {@code int} Size on the X axis
	 * @param pY                {@code int} Size on the Y axis
	 * @param pZ                {@code int} Size on the Z axis
	 * @param predicate         {@link BiPredicate} Predicate for the ring
	 * @param state             {@link class_2680} The state for the hologram
	 * @param holePredicate     {@link BiPredicate} Predicate for the hole
	 * @param holeHologramState {@link class_2680} The hole state for the hologram
	 * @return {@link MultiblockWriter} This. Useful for chaining
	 */
	default MultiblockWriter ring(class_2350.class_2351 through, int pX, int pY, int pZ, BiPredicate<class_1922, class_2338> predicate, class_2680 state, BiPredicate<class_1922, class_2338> holePredicate, class_2680 holeHologramState) {
		if (holePredicate == null) {
			holePredicate = predicate.negate();
		}

		if (holeHologramState == null) {
			holeHologramState = class_2246.field_10124.method_9564();
		}

		if (through == class_2350.class_2351.field_11048) {
			for (int y = 0; y < pY; y++) {
				for (int z = 0; z < pZ; z++) {
					if ((y == 0 || y == (pY - 1)) || (z == 0 || z == (pZ - 1))) {
						add(pX, y, z, predicate, state);
					} else {
						add(pX, y, z, holePredicate, holeHologramState);
					}
				}
			}
		} else if (through == class_2350.class_2351.field_11052) {
			for (int x = 0; x < pX; x++) {
				for (int z = 0; z < pZ; z++) {
					if ((x == 0 || x == (pX - 1)) || (z == 0 || z == (pZ - 1))) {
						add(x, pY, z, predicate, state);
					} else {
						add(x, pY, z, holePredicate, holeHologramState);
					}
				}
			}
		} else if (through == class_2350.class_2351.field_11051) {
			for (int x = 0; x < pX; x++) {
				for (int y = 0; y < pY; y++) {
					if ((x == 0 || x == (pX - 1)) || (y == 0 || y == (pY - 1))) {
						add(x, y, pZ, predicate, state);
					} else {
						add(x, y, pZ, holePredicate, holeHologramState);
					}
				}
			}
		}

		return this;
	}

	default MultiblockWriter ringWithAir(class_2350.class_2351 through, int x, int y, int z, BiPredicate<class_1922, class_2338> predicate, class_2680 state) {
		return ring(through, x, y, z, predicate, state, (view, pos) -> view.method_8320(pos).method_26204() == class_2246.field_10124, class_2246.field_10124.method_9564());
	}

	default MultiblockWriter add(int x, int y, int z, class_2680 state) {
		return this.add(x, y, z, (view, pos) -> view.method_8320(pos) == state, state);
	}

	default MultiblockWriter fill(int ax, int ay, int az, int bx, int by, int bz, class_2680 state) {
		return fill(ax, ay, az, bx, by, bz, (view, pos) -> view.method_8320(pos) == state, state);
	}

	default MultiblockWriter fill(int ax, int ay, int az, int bx, int by, int bz, class_2248 block) {
		return fill(ax, ay, az, bx, by, bz, (view, pos) -> view.method_8320(pos).method_27852(block), block.method_9564());
	}

	default MultiblockWriter ring(class_2350.class_2351 through, int x, int y, int z, class_2680 state, class_2680 holeState) {
		return ring(through, x, y, z, (view, pos) -> view.method_8320(pos) == state, state, (view, pos) -> view.method_8320(pos) == holeState, holeState);
	}

	default MultiblockWriter ringWithAir(class_2350.class_2351 through, int x, int y, int z, class_2680 state) {
		return ringWithAir(through, x, y, z, (view, pos) -> view.method_8320(pos) == state, state);
	}

	default MultiblockWriter ringWithAir(class_2350.class_2351 through, int x, int y, int z, class_2248 block) {
		return ringWithAir(through, x, y, z, (view, pos) -> view.method_8320(pos).method_27852(block), block.method_9564());
	}

	default MultiblockWriter translate(int offsetX, int offsetY, int offsetZ) {
		return (x, y, z, predicate, state) -> add(offsetX + x, offsetY + y, offsetZ + z, predicate, state);
	}

	default MultiblockWriter rotate() {
		return (x, y, z, predicate, state) -> add(-z, y, x, predicate, state);
	}

	default MultiblockWriter rotate(class_2350 direction) {
		MultiblockWriter w = this;

		switch (direction) {
			case field_11043:
				w = w.rotate();
			case field_11039:
				w = w.rotate();
			case field_11035:
				w = w.rotate();
		}

		return w;
	}

	/**
	 * A writer which prints the hologram to {@link System#out}
	 */
	record DebugWriter(MultiblockWriter writer) implements MultiblockWriter {

		@Override
		public MultiblockWriter add(int x, int y, int z, BiPredicate<class_1922, class_2338> predicate, class_2680 state) {
			System.out.printf("\t%d\t%d\t%d\t%s\n", x, y, z, state.method_26204());

			if (writer != null) {
				writer.add(x, y, z, predicate, state);
			}

			return this;
		}
	}

	/**
	 * A writer which verifies the positions of each block
	 */
	class MultiblockVerifier implements MultiblockWriter {
		private final class_2338 relative;
		private final class_1922 view;

		private boolean valid = true;

		public MultiblockVerifier(class_2338 relative, class_1922 view) {
			this.relative = relative;
			this.view = view;
		}

		public boolean isValid() {
			return valid;
		}

		@Override
		public MultiblockWriter add(int x, int y, int z, BiPredicate<class_1922, class_2338> predicate, class_2680 state) {
			if (valid) {
				valid = predicate.test(view, relative.method_10069(x, y, z));
			}

			return this;
		}
	}

	class MultiblockShapeFormer implements MultiblockWriter {
		private final class_2338 relative;
		private final Set<class_2338> pos = new HashSet<>();

		public MultiblockShapeFormer(class_2338 relative) {
			this.relative = relative;
		}

		@Override
		public MultiblockWriter add(int x, int y, int z, BiPredicate<class_1922, class_2338> predicate, class_2680 state) {
			pos.add(relative.method_10069(x, y, z));

			return this;
		}

		public Set<class_2338> getPos() {
			return pos;
		}
	}
}
