/*
 * This file is part of RebornCore, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2021 TeamReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.blockentity;

import org.apache.commons.lang3.Validate;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import reborncore.api.IListInfoProvider;
import reborncore.api.blockentity.IUpgrade;
import reborncore.api.blockentity.IUpgradeable;
import reborncore.api.blockentity.InventoryProvider;
import reborncore.api.recipe.IRecipeCrafterProvider;
import reborncore.common.blocks.BlockMachineBase;
import reborncore.common.fluid.FluidValue;
import reborncore.common.misc.world.ChunkEventListener;
import reborncore.common.misc.world.ChunkEventListeners;
import reborncore.common.network.NetworkManager;
import reborncore.common.network.clientbound.CustomDescriptionPayload;
import reborncore.common.recipes.IUpgradeHandler;
import reborncore.common.recipes.RecipeCrafter;
import reborncore.common.util.RebornInventory;
import reborncore.common.util.Tank;

import static reborncore.RebornCore.LOGGER;

import java.util.*;
import net.minecraft.class_11362;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_124;
import net.minecraft.class_1263;
import net.minecraft.class_1278;
import net.minecraft.class_1309;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1923;
import net.minecraft.class_1937;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2487;
import net.minecraft.class_2561;
import net.minecraft.class_2586;
import net.minecraft.class_2591;
import net.minecraft.class_2622;
import net.minecraft.class_2680;
import net.minecraft.class_5558;
import net.minecraft.class_7225;
import net.minecraft.class_8942;

/**
 * Created by modmuss50 on 04/11/2016.
 */
public class MachineBaseBlockEntity extends class_2586 implements class_5558<MachineBaseBlockEntity>, IUpgradeable, IUpgradeHandler, IListInfoProvider, class_1263, class_1278, RedstoneConfigurable, ChunkEventListener {

	public RebornInventory<MachineBaseBlockEntity> upgradeInventory = new RebornInventory<>(getUpgradeSlotCount(), "upgrades", 1, this, (slotID, stack, face, direction, blockEntity) -> true);
	private SlotConfiguration slotConfiguration;
	public FluidConfiguration fluidConfiguration;
	private RedstoneConfiguration redstoneConfiguration;
	private final List<RedstoneConfiguration.Element> redstoneElements;

	public boolean renderMultiblock = false;

	private boolean shapeValid = false;
	private boolean needsRematch = false;
	@Nullable
	private Set<class_2338> shape = null;

	private final static int syncCoolDown = 20;
	private boolean markSync = false;
	private int tickTime = 0;

	public static double SPEED_CAP = 0.99;

	/**
	 * <p>
	 *  This is used to change the speed of the crafting operation.
	 * <p/>
	 *
	 * <ul>
	 * 	<li>0 = Normal speed</li>
	 * 	<li>0.2 = 20% increase</li>
	 * 	<li>0.75 = 75% increase</li>
	 * </ul>
	 */
	double speedMultiplier = 0;
	/**
	 * <p>
	 *  This is used to change the power of the crafting operation.
	 * <p/>
	 *
	 * <ul>
	 *  <li>1 = Normal power</li>
	 *  <li>1.2 = 20% increase</li>
	 *  <li>1.75 = 75% increase</li>
	 *  <li>5 = 500% increase</li>
	 * </ul>
	 */
	double powerMultiplier = 1;
	/**
	 * <p>
	 *  This is used to change the sound of the crafting operation.
	 * <p/>
	 */
	boolean muffled = false;

	public MachineBaseBlockEntity(class_2591<?> type, class_2338 pos, class_2680 state) {
		super(type, pos, state);
		redstoneConfiguration = new RedstoneConfiguration();
		redstoneElements = RedstoneConfiguration.getValidElements(this);
	}

	public void rematch() {
		MultiblockWriter.MultiblockVerifier verifier = new MultiblockWriter.MultiblockVerifier(method_11016(), method_10997());
		writeMultiblock(verifier.rotate(getFacing().method_10153()));
		shapeValid = verifier.isValid();
		needsRematch = false;
	}

	public boolean isShapeValid() {
		return shapeValid;
	}

	public void setShapeValid(boolean shapeValid) {
		this.shapeValid = shapeValid;
	}

	public void link() {
		if (needsRematch) {
			rematch();
			syncWithAll();
		}
	}

	public void unlink() {
		if (shape != null) {
			unregisterListeners(field_11863);
			shape = null;
		}
	}

	public void registerMultiblockVerify() {
		MultiblockWriter.MultiblockShapeFormer writer = new MultiblockWriter.MultiblockShapeFormer(method_11016());
		writeMultiblock(writer.rotate(getFacing().method_10153()));

		shape = writer.getPos();
		registerListeners(field_11863);
		needsRematch = true;
	}

	public Set<class_1923> getSpannedChunks() {
		Set<class_1923> spannedChunks = new HashSet<>();

		assert shape != null;
		for (class_2338 pos : shape) {
			spannedChunks.add(new class_1923(pos));
		}

		return spannedChunks;
	}

	public void registerListeners(class_1937 world) {
		for (class_1923 chunkPos : getSpannedChunks()) {
			ChunkEventListeners.listeners.add(world, chunkPos, this);
		}
	}

	public void unregisterListeners(class_1937 world) {
		for (class_1923 chunkPos : getSpannedChunks()) {
			ChunkEventListeners.listeners.remove(world, chunkPos, this);
		}
	}

	@Override
	public void onBlockUpdate(class_2338 pos) {
		if (shape != null && shape.contains(pos)) {
			needsRematch = true;
		}
	}

	@Override
	public void onUnloadChunk() {
		needsRematch = true;
	}

	@Override
	public void onLoadChunk() {
		needsRematch = true;
	}

	@Override
	public final void method_11012() {
		super.method_11012();

		if (!field_11863.method_8608()) {
			unlink();
		}
	}

	private void syncIfNecessary(){
		if (this.markSync && this.tickTime % syncCoolDown == 0) {
			this.markSync = false;
			if (field_11863 == null || field_11863.method_8608()) { return; }
			NetworkManager.sendToTracking(new CustomDescriptionPayload(this.field_11867, this.method_38244(field_11863.method_30349())), this);
		}
	}

	public void writeMultiblock(MultiblockWriter writer) {}

	public boolean hasMultiblock() {
		return false;
	}

	public void syncWithAll() {
		this.markSync = true;
	}

	public void onLoad() {
		if (slotConfiguration == null) {
			if (getOptionalInventory().isPresent()) {
				slotConfiguration = new SlotConfiguration(getOptionalInventory().get());
			}
		}
		if (getTank() != null) {
			if (fluidConfiguration == null) {
				fluidConfiguration = new FluidConfiguration();
			}
		}
		if (hasMultiblock() && field_11863 != null && !field_11863.method_8608()) {
			registerMultiblockVerify();
		}
	}

	@Nullable
	@Override
	public class_2622 method_38235() {
		return class_2622.method_38585(this);
	}

	@Override
	public class_2487 method_16887(class_7225.class_7874 registryLookup) {
		class_2487 compound;
		try (class_8942.class_11340 logging = new class_8942.class_11340(method_71402(), LOGGER)) {
			class_11362 view = class_11362.method_71459(logging, registryLookup);
			super.method_11007(view);
			method_11007(view);
			compound = view.method_71475();
		}
		return compound;
	}

	@Override
	public void tick(class_1937 world, class_2338 pos, class_2680 state, MachineBaseBlockEntity blockEntity) {
		if (tickTime == 0) {
			onLoad();
		}
		tickTime++;
		@Nullable
		RecipeCrafter crafter = null;
		if (getOptionalCrafter().isPresent()) {
			crafter = getOptionalCrafter().get();
		}
		if (canBeUpgraded()) {
			resetUpgrades();
			for (int i = 0; i < getUpgradeSlotCount(); i++) {
				class_1799 stack = getUpgradeInventory().method_5438(i);
				if (!stack.method_7960() && stack.method_7909() instanceof IUpgrade) {
					((IUpgrade) stack.method_7909()).process(this, this, stack);
				}
			}
			afterUpgradesApplication();
		}
		if (world == null || world.method_8608()) {
			return;
		}

		link();

		if (crafter != null && isActive(RedstoneConfiguration.Element.RECIPE_PROCESSING)) {
			crafter.updateEntity();
		}
		if (slotConfiguration != null && isActive(RedstoneConfiguration.Element.ITEM_IO)) {
			slotConfiguration.update(this);
		}
		if (fluidConfiguration != null && isActive(RedstoneConfiguration.Element.FLUID_IO)) {
			fluidConfiguration.update(this);
		}
		syncIfNecessary();
	}

	public void resetUpgrades() {
		resetPowerMultiplier();
		resetSpeedMultiplier();
		resetMuffler();
	}

	protected void afterUpgradesApplication() {
	}

	public int getFacingInt() {
		class_2248 block = field_11863.method_8320(field_11867).method_26204();
		if (block instanceof BlockMachineBase) {
			return ((BlockMachineBase) block).getFacing(field_11863.method_8320(field_11867)).method_10146();
		}
		return 0;
	}

	public class_2350 getFacingEnum() {
		class_2248 block = field_11863.method_8320(field_11867).method_26204();
		if (block instanceof BlockMachineBase) {
			return ((BlockMachineBase) block).getFacing(field_11863.method_8320(field_11867));
		}
		return class_2350.field_11043;
	}

	public void setFacing(class_2350 enumFacing) {
		class_2248 block = field_11863.method_8320(field_11867).method_26204();
		if (block instanceof BlockMachineBase) {
			((BlockMachineBase) block).setFacing(enumFacing, field_11863, field_11867);
		}
	}

	public boolean isActive() {
		class_2248 block = field_11863.method_8320(field_11867).method_26204();
		if (block instanceof BlockMachineBase) {
			return field_11863.method_8320(field_11867).method_11654(BlockMachineBase.ACTIVE);
		}
		return false;
	}

	public Optional<RebornInventory<?>> getOptionalInventory() {
		if (this instanceof InventoryProvider inventory) {
			if (inventory.getInventory() == null) {
				return Optional.empty();
			}
			return Optional.of((RebornInventory<?>) inventory.getInventory());
		}
		return Optional.empty();
	}

	protected Optional<RecipeCrafter> getOptionalCrafter() {
		if (this instanceof IRecipeCrafterProvider crafterProvider) {
			if (crafterProvider.getRecipeCrafter() == null) {
				return Optional.empty();
			}
			return Optional.of(crafterProvider.getRecipeCrafter());
		}
		return Optional.empty();
	}

	@Override
	public void method_11014(class_11368 view) {
		super.method_11014(view);
		if (getOptionalInventory().isPresent()) {
			getOptionalInventory().get().read(view);
		}
		if (getOptionalCrafter().isPresent()) {
			getOptionalCrafter().get().read(view);
		}
		view.method_71420("slotConfig").ifPresentOrElse(config -> {
			slotConfiguration = new SlotConfiguration(config);
		}, () -> {
			if (getOptionalInventory().isPresent()) {
				slotConfiguration = new SlotConfiguration(getOptionalInventory().get());
			}
		});
		view.method_71420("fluidConfig").ifPresent(config -> {
			fluidConfiguration = new FluidConfiguration(config);
		});
		redstoneConfiguration = view.method_71426("redstoneConfig", RedstoneConfiguration.CODEC.codec()).orElseGet(RedstoneConfiguration::new);
		upgradeInventory.read(view, "Upgrades");
	}

	@Override
	public void method_11007(class_11372 view) {
		super.method_11007(view);
		if (getOptionalInventory().isPresent()) {
			getOptionalInventory().get().write(view);
		}
		if (getOptionalCrafter().isPresent()) {
			getOptionalCrafter().get().write(view);
		}
		if (slotConfiguration != null) {
			slotConfiguration.write(view.method_71461("slotConfig"));
		}
		if (fluidConfiguration != null) {
			fluidConfiguration.write(view.method_71461("fluidConfig"));
		}
		upgradeInventory.write(view, "Upgrades");
		view.method_71468("redstoneConfig", RedstoneConfiguration.CODEC.codec(), redstoneConfiguration);
	}

	// Inventory end

	@Override
	public class_1263 getUpgradeInventory() {
		return upgradeInventory;
	}

	@Override
	public int getUpgradeSlotCount() {
		return 4;
	}

	public class_2350 getFacing() {
		return getFacingEnum();
	}

	@Override
	public void resetSpeedMultiplier() {
		speedMultiplier = 0;
	}

	@Override
	public double getSpeedMultiplier() {
		return speedMultiplier;
	}

	@Override
	public void addPowerMultiplier(double amount) {
		powerMultiplier = powerMultiplier * (1f + amount);
	}

	@Override
	public void resetPowerMultiplier() {
		powerMultiplier = 1;
	}

	@Override
	public double getPowerMultiplier() {
		return powerMultiplier;
	}

	@Override
	public long getEuPerTick(long baseEu) {
		return (long) (baseEu * powerMultiplier);
	}

	@Override
	public void addSpeedMultiplier(double amount) {
		if (speedMultiplier + amount <= SPEED_CAP) {
			speedMultiplier += amount;
		} else {
			speedMultiplier = SPEED_CAP;
		}
	}

	@Override
	public void muffle() {
		muffled = true;
	}

	@Override
	public void resetMuffler() {
		muffled = false;
	}

	@Override
	public boolean isMuffled() {
		return muffled;
	}

	public boolean hasSlotConfig() {
		return true;
	}

	@Nullable
	public Tank getTank() {
		return null;
	}

	public boolean showTankConfig() {
		return getTank() != null;
	}

	//The amount of ticks between a slot transfer attempt, less is faster
	public int slotTransferSpeed() {
		return 4;
	}

	//The amount of fluid transferred each tick buy the fluid config
	public FluidValue fluidTransferAmount() {
		return FluidValue.BUCKET_QUARTER;
	}

	@Override
	public void addInfo(List<class_2561> info, boolean isReal, boolean hasData) {
		if (hasData) {
			if (getOptionalInventory().isPresent()) {
				info.add(class_2561.method_43470(class_124.field_1065 + "" + getOptionalInventory().get().getContents() + class_124.field_1080 + " items"));
			}
			if (!upgradeInventory.method_5442()) {
				info.add(class_2561.method_43470(class_124 .field_1065 + "" + upgradeInventory.getContents() + class_124 .field_1080 + " upgrades"));
			}
		}
	}

	public class_2248 getBlockType(){
		return field_11863.method_8320(field_11867).method_26204();
	}

	@Override
	public int method_5439() {
		if(getOptionalInventory().isPresent()){
			return getOptionalInventory().get().method_5439();
		}
		return 0;
	}

	@Override
	public boolean method_5442() {
		if(getOptionalInventory().isPresent()){
			return getOptionalInventory().get().method_5442();
		}
		return true;
	}

	@Override
	public class_1799 method_5438(int i) {
		if(getOptionalInventory().isPresent()){
			return getOptionalInventory().get().method_5438(i);
		}
		return class_1799.field_8037;
	}

	@Override
	public class_1799 method_5434(int i, int i1) {
		if(getOptionalInventory().isPresent()){
			return getOptionalInventory().get().method_5434(i, i1);
		}
		return class_1799.field_8037;
	}

	@Override
	public class_1799 method_5441(int i) {
		if(getOptionalInventory().isPresent()){
			return getOptionalInventory().get().method_5441(i);
		}
		return class_1799.field_8037;
	}

	@Override
	public void method_5447(int i, class_1799 itemStack) {
		if(getOptionalInventory().isPresent()){
			getOptionalInventory().get().method_5447(i, itemStack);
		}
	}

	@Override
	public boolean method_5443(class_1657 playerEntity) {
		if(getOptionalInventory().isPresent()){
			return getOptionalInventory().get().method_5443(playerEntity);
		}
		return false;
	}

	@Override
	public boolean method_5437(int slot, class_1799 stack) {
		if (slotConfiguration == null) {
			return false;
		}
		SlotConfiguration.SlotConfigHolder slotConfigHolder = slotConfiguration.getSlotDetails(slot);
		if (slotConfigHolder.filter() && getOptionalCrafter().isPresent()) {
			RecipeCrafter crafter = getOptionalCrafter().get();
			if (!crafter.isStackValidInput(stack)) {
				return false;
			}
		}
		return true;
	}

	@Override
	public void method_5448() {
		if(getOptionalInventory().isPresent()){
			getOptionalInventory().get().method_5448();
		}
	}

	@NotNull
	public SlotConfiguration getSlotConfiguration() {
		Validate.notNull(slotConfiguration, "slotConfiguration cannot be null");
		return slotConfiguration;
	}

	@Override
	public int[] method_5494(class_2350 side) {
		if(slotConfiguration == null){
			return new int[]{}; // I think should be ok, if needed this can return all the slots
		}
		return slotConfiguration.getSlotsForSide(side).stream()
			.filter(Objects::nonNull)
			.filter(slotConfig -> slotConfig.getSlotIO().ioConfig != SlotConfiguration.ExtractConfig.NONE)
			.mapToInt(SlotConfiguration.SlotConfig::getSlotID).toArray();
	}

	@Override
	public boolean method_5492(int index, class_1799 stack, @Nullable class_2350 direction) {
		if(direction == null || slotConfiguration == null){
			return false;
		}
		SlotConfiguration.SlotConfigHolder slotConfigHolder = slotConfiguration.getSlotDetails(index);
		SlotConfiguration.SlotConfig slotConfig = slotConfigHolder.getSideDetail(direction);
		if (slotConfig.getSlotIO().ioConfig.isInsert()) {
			if (slotConfigHolder.filter() && getOptionalCrafter().isPresent()) {
				RecipeCrafter crafter = getOptionalCrafter().get();
				return crafter.isStackValidInput(stack);
			}
			return slotConfig.getSlotIO().getIoConfig().isInsert();
		}
		return false;
	}

	@Override
	public boolean method_5493(int index, class_1799 stack, class_2350 direction) {
		if (slotConfiguration == null) {
			return false;
		}
		SlotConfiguration.SlotConfigHolder slotConfigHolder = slotConfiguration.getSlotDetails(index);
		SlotConfiguration.SlotConfig slotConfig = slotConfigHolder.getSideDetail(direction);
		return slotConfig.getSlotIO().ioConfig.isExtract();
	}

	public void onBreak(class_1937 world, class_1657 playerEntity, class_2338 blockPos, class_2680 blockState){

	}

	public void onPlace(class_1937 worldIn, class_2338 pos, class_2680 state, class_1309 placer, class_1799 stack){

	}

	public RedstoneConfiguration getRedstoneConfiguration() {
		return redstoneConfiguration;
	}

	public void setRedstoneConfiguration(RedstoneConfiguration redstoneConfiguration) {
		this.redstoneConfiguration = redstoneConfiguration;
	}

	@Override
	public boolean isActive(RedstoneConfiguration.Element element) {
		return redstoneConfiguration.isActive(element, this);
	}

	public List<RedstoneConfiguration.Element> getRedstoneElements() {
		return redstoneElements;
	}
}
