/*
 * This file is part of RebornCore, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2021 TeamReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.blockentity;

import io.netty.buffer.ByteBuf;
import net.fabricmc.fabric.api.transfer.v1.fluid.FluidStorage;
import net.fabricmc.fabric.api.transfer.v1.fluid.FluidVariant;
import net.fabricmc.fabric.api.transfer.v1.storage.Storage;
import net.fabricmc.fabric.api.transfer.v1.storage.StorageUtil;
import net.minecraft.class_11368;
import net.minecraft.class_11372;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_9135;
import net.minecraft.class_9139;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import reborncore.common.util.NBTSerializable;

import java.util.*;

public class FluidConfiguration implements NBTSerializable {
	private static final class_9139<ByteBuf, Map<class_2350, FluidConfig>> SIDE_MAP_PACKET_CODEC = class_9135.method_56377(
		HashMap::new,
		class_2350.field_48450,
		FluidConfig.PACKET_CODEC
	);
	public static final class_9139<ByteBuf, FluidConfiguration> PACKET_CODEC = class_9139.method_56436(
		SIDE_MAP_PACKET_CODEC, FluidConfiguration::getSideMap,
		class_9135.field_48547, FluidConfiguration::autoInput,
		class_9135.field_48547, FluidConfiguration::autoOutput,
		FluidConfiguration::new
	);

	Map<class_2350, FluidConfig> sideMap;
	boolean input, output;

	public FluidConfiguration() {
		sideMap = new HashMap<>();
		Arrays.stream(class_2350.values()).forEach(facing -> sideMap.put(facing, new FluidConfig(facing)));
	}

	public FluidConfiguration(class_11368 view) {
		sideMap = new HashMap<>();
		read(view);
	}

	private FluidConfiguration(Map<class_2350, FluidConfig> sideMap, boolean input, boolean output) {
		this.sideMap = sideMap;
		this.input = input;
		this.output = output;
	}

	public FluidConfig getSideDetail(class_2350 side) {
		if (side == null) {
			return sideMap.get(class_2350.field_11043);
		}
		return sideMap.get(side);
	}

	public List<FluidConfig> getAllSides() {
		return new ArrayList<>(sideMap.values());
	}

	public Map<class_2350, FluidConfig> getSideMap() {
		return sideMap;
	}

	public void updateFluidConfig(FluidConfig config) {
		FluidConfig toEdit = sideMap.get(config.side);
		toEdit.ioConfig = config.ioConfig;
	}

	public void update(MachineBaseBlockEntity machineBase) {
		if (!input && !output) {
			return;
		}
		if (machineBase.getTank() == null || machineBase.method_10997().method_8510() % machineBase.slotTransferSpeed() != 0) {
			return;
		}
		for (class_2350 facing : class_2350.values()) {
			FluidConfig fluidConfig = getSideDetail(facing);
			if (fluidConfig == null || !fluidConfig.getIoConfig().isEnabled()) {
				continue;
			}

			@Nullable
			Storage<FluidVariant> tank = getTank(machineBase, facing);
			if (autoInput() && fluidConfig.getIoConfig().isInsert()) {
				StorageUtil.move(tank, machineBase.getTank(), fv -> true, machineBase.fluidTransferAmount().getRawValue(), null);
			}
			if (autoOutput() && fluidConfig.getIoConfig().isExtract()) {
				StorageUtil.move(machineBase.getTank(), tank, fv -> true, machineBase.fluidTransferAmount().getRawValue(), null);
			}
		}
	}

	@Nullable
	private Storage<FluidVariant> getTank(MachineBaseBlockEntity machine, class_2350 facing) {
		class_2338 pos = machine.method_11016().method_10093(facing);
		return FluidStorage.SIDED.find(machine.method_10997(), pos, facing.method_10153());
	}

	public boolean autoInput() {
		return input;
	}

	public boolean autoOutput() {
		return output;
	}

	public void setInput(boolean input) {
		this.input = input;
	}

	public void setOutput(boolean output) {
		this.output = output;
	}

	@Override
	public void write(class_11372 view) {
		Arrays.stream(class_2350.values()).forEach(facing -> sideMap.get(facing).write(view.method_71461("side_" + facing.ordinal())));
		view.method_71472("input", input);
		view.method_71472("output", output);
	}

	@Override
	public void read(@NotNull class_11368 view) {
		sideMap.clear();
		Arrays.stream(class_2350.values()).forEach(facing -> {
			view.method_71420("side_" + facing.ordinal()).ifPresent(config -> {
				sideMap.put(facing, new FluidConfig(config));
			});
		});
		input = view.method_71433("input", false);
		output = view.method_71433("output", false);
	}

	public static class FluidConfig implements NBTSerializable {
		public static final class_9139<ByteBuf, FluidConfig> PACKET_CODEC = class_9139.method_56435(
			class_2350.field_48450, FluidConfig::getSide,
			ExtractConfig.PACKET_CODEC, FluidConfig::getIoConfig,
			FluidConfig::new
		);

		class_2350 side;
		FluidConfiguration.ExtractConfig ioConfig;

		public FluidConfig(class_2350 side) {
			this.side = side;
			this.ioConfig = ExtractConfig.ALL;
		}

		public FluidConfig(class_2350 side, FluidConfiguration.ExtractConfig ioConfig) {
			this.side = side;
			this.ioConfig = ioConfig;
		}

		public FluidConfig(class_11368 view) {
			read(view);
		}

		public class_2350 getSide() {
			return side;
		}

		public ExtractConfig getIoConfig() {
			return ioConfig;
		}

		@Override
		public void write(class_11372 view) {
			view.method_71465("side", side.ordinal());
			view.method_71465("config", ioConfig.ordinal());
		}

		@Override
		public void read(@NotNull class_11368 view) {
			side = class_2350.values()[view.method_71424("side", 0)];
			ioConfig = FluidConfiguration.ExtractConfig.values()[view.method_71424("config", 0)];
		}
	}

	public enum ExtractConfig {
		NONE(false, false),
		INPUT(false, true),
		OUTPUT(true, false),
		ALL(true, true);

		public static final class_9139<ByteBuf, ExtractConfig> PACKET_CODEC = class_9135.field_49675
			.method_56432(integer -> ExtractConfig.values()[integer], Enum::ordinal);

		boolean extract;
		boolean insert;

		ExtractConfig(boolean extract, boolean insert) {
			this.extract = extract;
			this.insert = insert;
		}

		public boolean isExtract() {
			return extract;
		}

		public boolean isInsert() {
			return insert;
		}

		public boolean isEnabled() {
			return extract || insert;
		}

		public FluidConfiguration.ExtractConfig getNext() {
			int i = this.ordinal() + 1;
			if (i >= FluidConfiguration.ExtractConfig.values().length) {
				i = 0;
			}
			return FluidConfiguration.ExtractConfig.values()[i];
		}
	}
}
