/*
 * This file is part of RebornCore, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2021 TeamReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common;

import com.google.common.collect.ImmutableList;
import com.mojang.brigadier.Command;
import com.mojang.brigadier.CommandDispatcher;
import com.mojang.brigadier.arguments.StringArgumentType;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.suggestion.SuggestionProvider;
import net.fabricmc.fabric.api.command.v2.CommandRegistrationCallback;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_2168;
import net.minecraft.class_2170;
import net.minecraft.class_2172;
import net.minecraft.class_2186;
import net.minecraft.class_2287;
import net.minecraft.class_2561;
import net.minecraft.class_2806;
import net.minecraft.class_3215;
import net.minecraft.class_3218;
import net.minecraft.class_3222;
import net.minecraft.class_7157;
import net.minecraft.class_7923;
import reborncore.common.network.NetworkManager;
import reborncore.common.network.clientbound.QueueItemStacksPayload;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import static com.mojang.brigadier.arguments.IntegerArgumentType.getInteger;
import static com.mojang.brigadier.arguments.IntegerArgumentType.integer;
import static com.mojang.brigadier.arguments.StringArgumentType.word;
import static net.minecraft.class_2170.method_9244;
import static net.minecraft.class_2170.method_9247;

public class RebornCoreCommands {

	private final static ExecutorService EXECUTOR_SERVICE = Executors.newSingleThreadExecutor();
	private final static SuggestionProvider<class_2168> MOD_SUGGESTIONS = (context, builder) ->
			class_2172.method_9264(FabricLoader.getInstance().getAllMods().stream().map(modContainer -> modContainer.getMetadata().getId()), builder);

	public static void setup() {
		CommandRegistrationCallback.EVENT.register((RebornCoreCommands::addCommands));
	}

	private static void addCommands(CommandDispatcher<class_2168> dispatcher, class_7157 registryAccess, class_2170.class_5364 environment) {
		dispatcher.register(
				method_9247("reborncore")

					.then(
						method_9247("generate")
							.requires(source -> source.method_9259(3))
							.then(method_9244("size", integer())
									.executes(RebornCoreCommands::generate)
							)
					)

					.then(
						method_9247("flyspeed")
							.requires(source -> source.method_9259(3))
							.then(method_9244("speed", integer(1, 10))
									.executes(ctx -> flySpeed(ctx, ImmutableList.of(ctx.getSource().method_44023())))
									.then(class_2170.method_9244("players", class_2186.method_9308())
											.executes(ctx -> flySpeed(ctx, class_2186.method_9312(ctx, "players")))
									)
							)
					)

					.then(
						method_9247("render")
							.then(
								method_9247("mod")
									.then(
										method_9244("modid", word())
										.suggests(MOD_SUGGESTIONS)
										.executes(RebornCoreCommands::renderMod)
									)
							)
							.then(
								method_9247("item")
									.then(
										method_9244("item", class_2287.method_9776(registryAccess))
										.executes(RebornCoreCommands::itemRenderer)
									)
							)
							.then(
								method_9247("hand")
								.executes(RebornCoreCommands::handRenderer)
							)
					)
		);
	}

	private static int generate(CommandContext<class_2168> ctx) {
		final int size = getInteger(ctx, "size");

		final class_3218 world = ctx.getSource().method_9225();
		final class_3215 serverChunkManager = world.method_14178();
		final AtomicInteger completed = new AtomicInteger(0);

		for (int x = -(size / 2); x < size / 2; x++) {
			for (int z = -(size / 2); z < size / 2; z++) {
				final int chunkPosX = x;
				final int chunkPosZ = z;
				CompletableFuture.supplyAsync(() -> serverChunkManager.method_12121(chunkPosX, chunkPosZ, class_2806.field_12803, true), EXECUTOR_SERVICE)
						.whenComplete((chunk, throwable) -> {
									int max = (int) Math.pow(size, 2);
									ctx.getSource().method_9226(() -> class_2561.method_43470(String.format("Finished generating %d:%d (%d/%d %d%%)", chunk.method_12004().field_9181, chunk.method_12004().field_9180, completed.getAndIncrement(), max, completed.get() == 0 ? 0 : (int) ((completed.get() * 100.0f) / max))), true);
								}
						);
			}
		}
		return Command.SINGLE_SUCCESS;
	}

	private static int flySpeed(CommandContext<class_2168> ctx, Collection<class_3222> players) {
		final int speed = getInteger(ctx, "speed");
		players.stream()
				.peek(player -> player.method_31549().method_7248(speed / 20F))
				.forEach(class_3222::method_7355);

		return Command.SINGLE_SUCCESS;
	}

	private static int renderMod(CommandContext<class_2168> ctx) {
		String modid = StringArgumentType.getString(ctx, "modid");

		List<class_1799> list = class_7923.field_41178.method_10235().stream()
				.filter(identifier -> identifier.method_12836().equals(modid))
				.map(class_7923.field_41178::method_63535)
				.map(class_1799::new)
				.collect(Collectors.toList());

		queueRender(list, ctx);
		return Command.SINGLE_SUCCESS;
	}

	private static int itemRenderer(CommandContext<class_2168> ctx) {
		class_1792 item = class_2287.method_9777(ctx, "item").method_9785();
		queueRender(Collections.singletonList(new class_1799(item)), ctx);

		return Command.SINGLE_SUCCESS;
	}

	private static int handRenderer(CommandContext<class_2168> ctx) {
		queueRender(Collections.singletonList(ctx.getSource().method_44023().method_31548().method_7391()), ctx);

		return Command.SINGLE_SUCCESS;
	}

	private static void queueRender(List<class_1799> stacks, CommandContext<class_2168> ctx) {
		NetworkManager.sendToPlayer(new QueueItemStacksPayload(stacks), ctx.getSource().method_44023());
	}
}
