/*
 * This file is part of RebornCore, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2021 TeamReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.client.gui.config;

import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import net.fabricmc.fabric.api.client.networking.v1.ClientPlayNetworking;
import net.minecraft.class_11908;
import net.minecraft.class_1735;
import net.minecraft.class_1799;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_332;
import org.jetbrains.annotations.Nullable;
import org.lwjgl.glfw.GLFW;
import reborncore.client.ClientChatUtils;
import reborncore.client.gui.GuiBase;
import reborncore.client.gui.config.elements.ConfigSlotElement;
import reborncore.client.gui.config.elements.SlotType;
import reborncore.common.network.serverbound.SlotConfigSavePayload;
import reborncore.common.screen.BuiltScreenHandler;
import reborncore.common.screen.slot.BaseSlot;
import reborncore.common.util.Color;

import java.util.List;
import java.util.Objects;

public class SlotConfigGui extends GuiTab {
	private Int2ObjectMap<ConfigSlotElement> slotElementMap = new Int2ObjectOpenHashMap<>();

	@Nullable
	private ConfigSlotElement selectedSlot;

	public SlotConfigGui(GuiBase<?> guiBase) {
		super(guiBase);
	}

	@Override
	public String name() {
		return "reborncore.gui.tooltip.config_slots";
	}

	@Override
	public boolean enabled() {
		return machine.hasSlotConfig();
	}

	@Override
	public class_1799 stack() {
		return GuiBase.wrenchStack;
	}

	@Override
	public void open() {
		selectedSlot = null;
		slotElementMap.clear();

		BuiltScreenHandler container = guiBase.builtScreenHandler;
		for (class_1735 slot : container.field_7761) {
			if (guiBase.be != slot.field_7871) {
				continue;
			}

			ConfigSlotElement slotElement = new ConfigSlotElement(
				guiBase.getMachine().getOptionalInventory().get(),
				(BaseSlot) slot,
				SlotType.NORMAL,
				slot.field_7873 - guiBase.getGuiLeft() + 50,
				slot.field_7872 - guiBase.getGuiTop() - 25,
				guiBase,
				this::close
			);
			slotElementMap.put(slot.method_34266(), slotElement);
		}

	}

	@Override
	public void close() {
		selectedSlot = null;
	}

	@Override
	public void draw(class_332 drawContext, int x, int y) {
		BuiltScreenHandler container = guiBase.builtScreenHandler;
		for (class_1735 slot : container.field_7761) {
			if (guiBase.be != slot.field_7871) {
				continue;
			}
			Color color = new Color(255, 0, 0, 128);

			drawContext.method_25294(slot.field_7873 -1, slot.field_7872 -1, slot.field_7873 + 17, slot.field_7872 + 17, color.getColor());
		}

		if (selectedSlot != null) {
			selectedSlot.draw(drawContext, guiBase, x, y);
		}
	}

	@Override
	public boolean click(double mouseX, double mouseY, int mouseButton) {
		final BuiltScreenHandler screenHandler = Objects.requireNonNull(guiBase.builtScreenHandler);

		if (selectedSlot != null) {
			return selectedSlot.onClick(guiBase, mouseX, mouseY);
		} else {
			for (class_1735 slot : screenHandler.field_7761) {
				if (guiBase.be != slot.field_7871) {
					continue;
				}
				if (guiBase.isPointInRect(slot.field_7873, slot.field_7872, 18, 18, mouseX, mouseY)) {
					selectedSlot = slotElementMap.get(slot.method_34266());
					return true;
				}
			}
		}

		return false;
	}

	@Override
	public void mouseReleased(double mouseX, double mouseY, int state) {
		if (selectedSlot != null) {
			selectedSlot.mouseReleased(mouseX, mouseY, state);
		}
	}

	@Override
	public boolean keyPress(class_11908 key) {
		int keyCode = key.comp_4795();
		if (key.method_74240() && keyCode == GLFW.GLFW_KEY_C) {
			copyToClipboard();
			return true;
		} else if (key.method_74240() && keyCode == GLFW.GLFW_KEY_V) {
			pasteFromClipboard();
			return true;
		} else if (keyCode == GLFW.GLFW_KEY_ESCAPE && selectedSlot != null) {
			selectedSlot = null;
			return true;
		}
		return false;
	}

	@Override
	public List<String> getTips() {
		return List.of(
			"reborncore.gui.slotconfigtip.slot",
			"reborncore.gui.slotconfigtip.side1",
			"reborncore.gui.slotconfigtip.side2",
			"reborncore.gui.slotconfigtip.side3",
			"reborncore.gui.slotconfigtip.copy1",
			"reborncore.gui.slotconfigtip.copy2"
		);
	}

	private void copyToClipboard() {
		machine.getSlotConfiguration();
		class_310 client = class_310.method_1551();
		String json = machine.getSlotConfiguration().toJson(machine.getClass().getCanonicalName(), client.field_1687.method_30349());
		client.field_1774.method_1455(json);
		ClientChatUtils.addHudMessage(class_2561.method_43470("Slot configuration copied to clipboard"));
	}

	private void pasteFromClipboard() {
		machine.getSlotConfiguration();

		class_310 client = class_310.method_1551();
		String json = client.field_1774.method_1460();
		try {
			machine.getSlotConfiguration().readJson(json, machine.getClass().getCanonicalName(), client.field_1687.method_30349());
			ClientPlayNetworking.send(new SlotConfigSavePayload(machine.method_11016(), machine.getSlotConfiguration()));
			ClientChatUtils.addHudMessage(class_2561.method_43470("Slot configuration loaded from clipboard"));
		} catch (UnsupportedOperationException e) {
			ClientChatUtils.addHudMessage(class_2561.method_43470(e.getMessage()));
		}
	}

	public ConfigSlotElement getSelectedSlot() {
		return selectedSlot;
	}
}
