/*
 * This file is part of RebornCore, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2021 TeamReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.client.gui;

import com.google.common.collect.Lists;
import net.fabricmc.fabric.api.transfer.v1.client.fluid.FluidVariantRendering;
import net.minecraft.class_1058;
import net.minecraft.class_10799;
import net.minecraft.class_124;
import net.minecraft.class_1799;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_350;
import net.minecraft.class_3612;
import net.minecraft.class_437;
import net.minecraft.class_4730;
import net.minecraft.class_6382;
import reborncore.api.IListInfoProvider;
import reborncore.client.gui.config.GuiTab;
import reborncore.common.fluid.FluidUtils;
import reborncore.common.fluid.FluidValue;
import reborncore.common.fluid.container.FluidInstance;
import reborncore.common.powerSystem.PowerSystem;
import reborncore.common.util.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;

import static reborncore.client.gui.GuiSprites.drawSpriteStretched;

public class GuiBuilder {
	private static final class_2561 SPACE_TEXT = class_2561.method_43470(" ");
	@Deprecated
	public static final class_2960 GUI_ELEMENTS = class_2960.method_60655("reborncore", "textures/gui/guielements.png");
	private static final boolean EXPERIMENTAL_PROGRESS_BAR = false;

	public void drawDefaultBackground(class_332 drawContext, int x, int y, int width, int height) {
		drawContext.method_52706(class_10799.field_56883, GuiSprites.BACKGROUND.method_24147(), x, y, width, height);
	}

	public void drawPlayerSlots(class_332 drawContext, class_437 gui, int posX, int posY, boolean center) {
		if (center) {
			posX -= 81;
		}

		for (int y = 0; y < 3; y++) {
			for (int x = 0; x < 9; x++) {
				drawSlot(drawContext,posX + x * 18, posY + y * 18);
			}
		}

		for (int x = 0; x < 9; x++) {
			drawSlot(drawContext, posX + x * 18, posY + 58);
		}
	}

	public void drawSlot(class_332 drawContext,int posX, int posY) {
		drawSpriteStretched(drawContext, GuiSprites.SLOT, posX, posY, 18, 18);
	}

	public void drawText(class_332 drawContext, GuiBase<?> gui, class_2561 text, int x, int y, int color) {
		drawContext.method_51439(gui.method_64506(), text, x, y, color, false);
	}

	public void drawProgressBar(class_332 drawContext, GuiBase<?> gui, double progress, int x, int y) {
		drawContext.method_25290(class_10799.field_56883, GUI_ELEMENTS, x, y, 150, 18, 22, 15, 256, 256);
		int j = (int) (progress);
		if (j > 0) {
			drawContext.method_25290(class_10799.field_56883, GUI_ELEMENTS, x, y, 150, 34, j + 1, 15, 256, 256);
		}
	}

	public void drawOutputSlot(class_332 drawContext, int x, int y) {
		drawSpriteStretched(drawContext, GuiSprites.OUTPUT_SLOT, x, y, 26, 26);
	}

	/**
	 * Draws lock button in either locked or unlocked state
	 *
	 * @param gui    {@link GuiBase} The GUI to draw on
	 * @param x      {@code int} Top left corner where to place button
	 * @param y      {@code int} Top left corner where to place button
	 * @param mouseX {@code int} Mouse cursor position to check for tooltip
	 * @param mouseY {@code int} Mouse cursor position to check for tooltip
	 * @param layer  {@link GuiBase.Layer} The layer to draw on
	 * @param locked {@code boolean} Set to true if it is in locked state
	 */
	public void drawLockButton(class_332 drawContext, GuiBase<?> gui, int x, int y, int mouseX, int mouseY, GuiBase.Layer layer, boolean locked) {
		if (gui.hideGuiElements()) return;
		int x2 = x, y2 = y;
		if (layer == GuiBase.Layer.BACKGROUND) {
			x2 += gui.getGuiLeft();
			y2 += gui.getGuiTop();
		}

		drawSpriteStretched(drawContext, locked ? GuiSprites.BUTTON_LOCKED : GuiSprites.BUTTON_UNLOCKED, x2, y2, 20, 12);
		if (gui.isPointInRect(x, y, 20, 12, mouseX, mouseY)) {
			List<class_2561> list = new ArrayList<>();
			if (locked) {
				list.add(class_2561.method_43471("reborncore.gui.tooltip.unlock_items"));
			} else {
				list.add(class_2561.method_43471("reborncore.gui.tooltip.lock_items"));
			}
			drawContext.method_51434(gui.method_64506(), list, mouseX, mouseY);
		}
	}

	/**
	 * Draws hologram toggle button
	 *
	 * @param gui    {@link GuiBase} The GUI to draw on
	 * @param x      {@code int} Top left corner where to place button
	 * @param y      {@code int} Top left corner where to place button
	 * @param mouseX {@code int} Mouse cursor position to check for tooltip
	 * @param mouseY {@code int} Mouse cursor position to check for tooltip
	 * @param layer  {@link GuiBase.Layer} The layer to draw on
	 */
	public void drawHologramButton(class_332 drawContext, GuiBase<?> gui, int x, int y, int mouseX, int mouseY, GuiBase.Layer layer) {
		if (gui.isTabOpen()) return;
		boolean hasTooltip = gui.isPointInRect(x, y, 20, 12, mouseX, mouseY);
		if (layer == GuiBase.Layer.BACKGROUND) {
			x += gui.getGuiLeft();
			y += gui.getGuiTop();
		}
		if (gui.getMachine().renderMultiblock) {
			drawSpriteStretched(drawContext, GuiSprites.BUTTON_HOLOGRAM_ENABLED, x, y, 20, 12);
		} else {
			drawSpriteStretched(drawContext, GuiSprites.BUTTON_HOLOGRAM_DISABLED, x, y, 20, 12);
		}
		if (hasTooltip) {
			List<class_2561> list = new ArrayList<>();
			list.add(class_2561.method_43471("reborncore.gui.tooltip.hologram"));
			drawContext.method_51434(gui.method_64506(), list, mouseX, mouseY);
		}
	}

	/**
	 * Draws big horizontal bar for heat value
	 *
	 * @param gui   {@link GuiBase} The GUI to draw on
	 * @param x     {@code int} Top left corner where to place bar
	 * @param y     {@code int} Top left corner where to place bar
	 * @param value {@code int} Current heat value
	 * @param max   {@code int} Maximum heat value
	 * @param layer {@link GuiBase.Layer} The layer to draw on
	 */
	public void drawBigHeatBar(class_332 drawContext, GuiBase<?> gui, int x, int y, int value, int max, GuiBase.Layer layer) {
		if (gui.hideGuiElements()) return;
		if (layer == GuiBase.Layer.BACKGROUND) {
			x += gui.getGuiLeft();
			y += gui.getGuiTop();
		}
		drawContext.method_25290(class_10799.field_56883, GUI_ELEMENTS, x, y, 26, 218, 114, 18, 256, 256);
		if (value != 0) {
			int j = (int) ((double) value / (double) max * 106);
			if (j < 0) {
				j = 0;
			}
			drawContext.method_25290(class_10799.field_56883, GUI_ELEMENTS, x + 4, y + 4, 26, 246, j, 10, 256, 256);

			class_2561 text = class_2561.method_43470(String.valueOf(value))
					.method_10852(class_2561.method_43471("reborncore.gui.heat"));

			gui.drawCentredText(drawContext, text, y + 5, 0xFFFFFFFF, layer);
		}
	}

	/**
	 * Draws big horizontal blue bar
	 *
	 * @param gui    {@link GuiBase} The GUI to draw on
	 * @param x      {@code int} Top left corner where to place bar
	 * @param y      {@code int} Top left corner where to place bar
	 * @param value  {@code int} Current value
	 * @param max    {@code int} Maximum value
	 * @param mouseX {@code int} Mouse cursor position to check for tooltip
	 * @param mouseY {@code int} Mouse cursor position to check for tooltip
	 * @param suffix {@link String} String to put on the bar and tooltip after percentage value
	 * @param line2  {@link String} String to put into tooltip as a second line
	 * @param format {@link String} Formatted value to put on the bar
	 * @param layer  {@link GuiBase.Layer} The layer to draw on
	 */
	public void drawBigBlueBar(class_332 drawContext, GuiBase<?> gui, int x, int y, int value, int max, int mouseX, int mouseY, String suffix, class_2561 line2, String format, GuiBase.Layer layer) {
		if (gui.hideGuiElements()) return;
		if (layer == GuiBase.Layer.BACKGROUND) {
			x += gui.getGuiLeft();
			y += gui.getGuiTop();
		}
		int j = (int) ((double) value / (double) max * 106);
		if (j < 0) {
			j = 0;
		}
		drawContext.method_25290(class_10799.field_56883, GUI_ELEMENTS, x + 4, y + 4, 0, 236, j, 10, 256, 256);
		if (!suffix.equals("")) {
			suffix = " " + suffix;
		}
		gui.drawCentredText(drawContext, class_2561.method_43470(format).method_27693(suffix), y + 5, 0xFFFFFFFF, layer);
		if (gui.isPointInRect(x, y, 114, 18, mouseX, mouseY)) {
			int percentage = percentage(max, value);
			List<class_2561> list = new ArrayList<>();

			list.add(
					class_2561.method_43470(String.valueOf(value))
							.method_27692(class_124.field_1065)
							.method_27693("/")
							.method_27693(String.valueOf(max))
							.method_27693(suffix)
			);

			list.add(
					class_2561.method_43470(String.valueOf(percentage))
							.method_27692(StringUtils.getPercentageColour(percentage))
							.method_27693("%")
							.method_10852(
									class_2561.method_43471("reborncore.gui.tooltip.dsu_fullness")
											.method_27692(class_124.field_1080)
							)
			);

			list.add(line2);

			if (value > max) {
				list.add(
						class_2561.method_43470("Yo this is storing more than it should be able to")
								.method_27692(class_124.field_1080)
				);
				list.add(
						class_2561.method_43470("prolly a bug")
								.method_27692(class_124.field_1080)
				);
				list.add(
						class_2561.method_43470("pls report and tell how tf you did this")
								.method_27692(class_124.field_1080)
				);
			}
			if (layer == GuiBase.Layer.FOREGROUND) {
				mouseX -= gui.getGuiLeft();
				mouseY -= gui.getGuiTop();
			}
			drawContext.method_51434(gui.method_64506(), list, mouseX, mouseY);
		}
	}

	public void drawBigBlueBar(class_332 drawContext, GuiBase<?> gui, int x, int y, int value, int max, int mouseX, int mouseY, String suffix, GuiBase.Layer layer) {
		drawBigBlueBar(drawContext, gui, x, y, value, max, mouseX, mouseY, suffix, class_2561.method_43473(), Integer.toString(value), layer);

	}

	public void drawBigBlueBar(class_332 drawContext, GuiBase<?> gui, int x, int y, int value, int max, int mouseX, int mouseY, GuiBase.Layer layer) {
		drawBigBlueBar(drawContext, gui, x, y, value, max, mouseX, mouseY, "", class_2561.method_43473(), "", layer);
	}

	/**
	 * Shades GUI and draw gray bar on top of GUI
	 *
	 * @param gui   {@link GuiBase} The GUI to draw on
	 * @param layer {@link GuiBase.Layer} The layer to draw on
	 */
	public void drawMultiblockMissingBar(class_332 drawContext, GuiBase<?> gui, GuiBase.Layer layer) {
		if (gui.hideGuiElements()) return;
		int x = 0;
		int y = 4;
		if (layer == GuiBase.Layer.BACKGROUND) {
			x += gui.getGuiLeft();
			y += gui.getGuiTop();
		}

		drawContext.method_25296(x, y, x + 176, y + 20, 0x000000, 0xC0000000);
		drawContext.method_25296(x, y + 20, x + 176, y + 20 + 48, 0xC0000000, 0xC0000000);
		drawContext.method_25296(x, y + 68, x + 176, y + 70 + 20, 0xC0000000, 0x00000000);

		gui.drawCentredText(drawContext, class_2561.method_43471("reborncore.gui.missingmultiblock"), 43, 0xFFFFFFFF, layer);
	}

	/**
	 * Draws upgrade slots on the left side of machine GUI. Draws on the background
	 * level.
	 *
	 * @param gui {@link GuiBase} The GUI to draw on
	 * @param x   {@code int} Top left corner where to place slots
	 * @param y   {@code int} Top left corner where to place slots
	 */
	public void drawUpgrades(class_332 drawContext, GuiBase<?> gui, int x, int y) {
		drawSpriteStretched(drawContext, GuiSprites.UPGRADES, x, y, 24, 81);
	}

	/**
	 * Draws tab on the left side of machine GUI. Draws on the background level.
	 *
	 * @param gui   {@link GuiBase} The GUI to draw on
	 * @param x     {@code int} Top left corner where to place tab
	 * @param y     {@code int} Top left corner where to place tab
	 * @param stack {@link class_1799} Item to show as tab icon
	 */
	public void drawSlotTab(class_332 drawContext, GuiBase<?> gui, int x, int y, class_1799 stack) {
		drawSpriteStretched(drawContext, GuiSprites.SLOT_TAB, x, y, 24, 24);
		drawContext.method_51427(stack, x + 5, y + 4);
	}


	/**
	 * Draws Slot Configuration tips instead of player inventory
	 *
	 * @param gui    {@link GuiBase} The GUI to draw on
	 * @param x      {@code int} Top left corner where to place tips list
	 * @param y      {@code int} Top left corner where to place tips list
	 * @param mouseX {@code int} Mouse cursor position
	 * @param mouseY {@code int} Mouse cursor position
	 */
	public void drawSlotConfigTips(class_332 drawContext, GuiBase<?> gui, int x, int y, int mouseX, int mouseY, GuiTab guiTab) {
		List<class_2561> tips = guiTab.getTips().stream()
				.map(class_2561::method_43471)
				.collect(Collectors.toList());

		TipsListWidget explanation = new TipsListWidget(gui, gui.getScreenWidth() - 14, 76, y, 9 + 2, tips);
		explanation.method_46421(x - 81);
		explanation.method_44382(0);
		explanation.method_25394(drawContext, mouseX, mouseY, 1.0f);
	}

	private static class TipsListWidget extends class_350<TipsListWidget.TipsListEntry> {
		private final Theme theme;

		public TipsListWidget(GuiBase<?> gui, int width, int height, int top, int entryHeight, List<class_2561> tips) {
			super(gui.getMinecraft(), width, height, top, entryHeight);
			for (class_2561 tip : tips) {
				this.method_25321(new TipsListEntry(tip));
			}
			theme = gui.theme;
		}

		@Override
		public int method_25322() {
			return 162;
		}

		@Override
		public void method_25311(class_332 drawContext, int mouseX, int mouseY, float delta) {
			drawContext.method_25294(this.method_46426(), this.method_46427(), this.method_46426() + this.method_25368(), this.method_46427() + this.method_25364(), 0xff202020);
			super.method_25311(drawContext, mouseX, mouseY, delta);
		}

		@Override
		protected void method_47399(class_6382 builder) {
		}

		private class TipsListEntry extends class_350.class_351<TipsListWidget.TipsListEntry> {
			private final class_2561 tip;

			public TipsListEntry(class_2561 tip) {
				this.tip = tip;
			}

			@Override
			public void renderContent(class_332 drawContext, int mouseX, int mouseY, boolean hovered, float deltaTicks) {
				drawContext.method_65179(class_310.method_1551().field_1772, tip, getContentX(), getContentY(), method_25368(), theme.subtitleColor().comp_1971());
			}
		}
	}

	// TODO: change to double

	/**
	 * Draws energy output value and icon
	 *
	 * @param gui       {@link GuiBase} The GUI to draw on
	 * @param x         {@code int} Top left corner where to place energy output
	 * @param y         {@code int} Top left corner where to place energy output
	 * @param maxOutput {@code int} Energy output value
	 * @param layer     {@link GuiBase.Layer} The layer to draw on
	 */
	public void drawEnergyOutput(class_332 drawContext, GuiBase<?> gui, int x, int y, int maxOutput, GuiBase.Layer layer) {
		if (gui.hideGuiElements()) return;
		class_2561 text = class_2561.method_43470(PowerSystem.getLocalizedPowerNoSuffix(maxOutput))
				.method_10852(SPACE_TEXT)
				.method_27693(PowerSystem.ABBREVIATION)
				.method_27693(" ");

		int width = gui.method_64506().method_27525(text);
		gui.drawText(drawContext, text, x - width - 2, y + 5, 0xff000000, layer);
		if (layer == GuiBase.Layer.BACKGROUND) {
			x += gui.getGuiLeft();
			y += gui.getGuiTop();
		}
		drawContext.method_25290(class_10799.field_56883, GUI_ELEMENTS, x, y, 150, 91, 16, 16, 256, 256);
	}

	/**
	 * Draws progress arrow in direction specified.
	 *
	 * @param gui         {@link GuiBase} The GUI to draw on
	 * @param progress    {@code int} Current progress
	 * @param maxProgress {@code int} Maximum progress
	 * @param x           {@code int} Top left corner where to place progress arrow
	 * @param y           {@code int} Top left corner where to place progress arrow
	 * @param mouseX      {@code int} Mouse cursor position to check for tooltip
	 * @param mouseY      {@code int} Mouse cursor position to check for tooltip
	 * @param direction   {@link ProgressDirection} Direction of the progress arrow
	 * @param layer       {@link GuiBase.Layer} The layer to draw on
	 */
	public void drawProgressBar(class_332 drawContext, GuiBase<?> gui, int progress, int maxProgress, int x, int y, int mouseX, int mouseY, ProgressDirection direction, GuiBase.Layer layer) {
		if (gui.hideGuiElements()) return;
		if (layer == GuiBase.Layer.BACKGROUND) {
			x += gui.getGuiLeft();
			y += gui.getGuiTop();
		}

		drawSpriteStretched(drawContext, direction.baseSprite, x, y, direction.width, direction.height);
		int j = (int) ((double) progress / (double) maxProgress * 16);
		if (j < 0) {
			j = 0;
		}

		if (EXPERIMENTAL_PROGRESS_BAR) {
			switch (direction) {
				case RIGHT, LEFT -> drawSpriteStretched(drawContext, direction.overlaySprite, x, y, j, 10, direction.width, direction.height, gui);
				case UP, DOWN -> drawSpriteStretched(drawContext, direction.overlaySprite, x, y, 10, j, direction.width, direction.height, gui);
			}
		} else {
			switch (direction) {
				case RIGHT -> drawContext.method_25290(class_10799.field_56883, GUI_ELEMENTS, x, y, direction.xActive, direction.yActive, j, 10, 256, 256);
				case LEFT -> drawContext.method_25290(class_10799.field_56883, GUI_ELEMENTS, x + 16 - j, y, direction.xActive + 16 - j, direction.yActive, j, 10, 256, 256);
				case UP -> drawContext.method_25290(class_10799.field_56883, GUI_ELEMENTS, x, y + 16 - j, direction.xActive, direction.yActive + 16 - j, 10, j, 256, 256);
				case DOWN -> drawContext.method_25290(class_10799.field_56883, GUI_ELEMENTS, x, y, direction.xActive, direction.yActive, 10, j, 256, 256);
			}
		}

		final class_1058 sprite = GuiBase.getSprite(direction.baseSprite);

		if (gui.isPointInRect(x, y, direction.width, direction.height, mouseX, mouseY)) {
			int percentage = percentage(maxProgress, progress);
			List<class_2561> list = new ArrayList<>();
			list.add(
					class_2561.method_43470(String.valueOf(percentage))
							.method_27692(StringUtils.getPercentageColour(percentage))
							.method_27693("%")
			);
			drawContext.method_51434(gui.method_64506(), list, mouseX, mouseY);
		}
	}

	/**
	 * Draws multi-energy bar
	 *
	 * @param gui             {@link GuiBase} The GUI to draw on
	 * @param x               {@code int} Top left corner where to place energy bar
	 * @param y               {@code int} Top left corner where to place energy bar
	 * @param energyStored    {@code long} Current amount of energy
	 * @param maxEnergyStored {@code long} Maximum amount of energy
	 * @param mouseX          {@code int} Mouse cursor position to check for tooltip
	 * @param mouseY          {@code int} Mouse cursor position to check for tooltip
	 * @param buttonID        {@code int} Button ID used to switch energy systems
	 * @param layer           {@link GuiBase.Layer} The layer to draw on
	 */
	public void drawMultiEnergyBar(class_332 drawContext, GuiBase<?> gui, int x, int y, long energyStored, long maxEnergyStored, int mouseX,
								int mouseY, int buttonID, GuiBase.Layer layer) {
		if (gui.hideGuiElements()) return;
		if (layer == GuiBase.Layer.BACKGROUND) {
			x += gui.getGuiLeft();
			y += gui.getGuiTop();
		}

		drawSpriteStretched(drawContext, GuiSprites.POWER_BAR_BASE, x, y, 14, 50);

		int barHeight = 48;
		int draw = (int) ((double) energyStored / (double) maxEnergyStored * (barHeight));
		if (energyStored > maxEnergyStored) {
			draw = barHeight;
		}
		drawSpriteStretched(drawContext, GuiSprites.POWER_BAR_OVERLAY, x + 1, y + 49 - draw, 12, draw, 12, 48);

		int percentage = percentage(maxEnergyStored, energyStored);
		if (gui.isPointInRect(x + 1, y + 1, 11, 48, mouseX, mouseY)) {
			List<class_2561> list = Lists.newArrayList();
			boolean hasShift = class_310.method_1551().method_74187();
			if (hasShift) {
				list.add(
						class_2561.method_43470(PowerSystem.getLocalizedPowerFullNoSuffix(energyStored))
								.method_27692(class_124.field_1065)
								.method_27693("/")
								.method_27693(PowerSystem.getLocalizedPowerFull(maxEnergyStored))
				);
			} else {
				list.add(
						class_2561.method_43470(PowerSystem.getLocalizedPowerNoSuffix(energyStored))
								.method_27692(class_124.field_1065)
								.method_27693("/")
								.method_27693(PowerSystem.getLocalizedPower(maxEnergyStored))
				);
			}
			list.add(
					StringUtils.getPercentageText(percentage)
							.method_10852(SPACE_TEXT)
							.method_10852(
									class_2561.method_43471("reborncore.gui.tooltip.power_charged")
											.method_27692(class_124.field_1080)
							)
			);

			if (gui.be instanceof IListInfoProvider) {
				if (hasShift) {
					((IListInfoProvider) gui.be).addInfo(list, true, true);
				} else {
					list.add(class_2561.method_43473());

					list.add(
							class_2561.method_43470("Shift")
									.method_27692(class_124.field_1078)
									.method_10852(SPACE_TEXT)
									.method_27692(class_124.field_1080)
									.method_10852(class_2561.method_43471("reborncore.gui.tooltip.power_moreinfo"))
					);
				}
			}
			drawContext.method_51434(gui.method_64506(), list, mouseX, mouseY);
		}
	}

	/**
	 * Draws tank and fluid inside it
	 *
	 * @param gui         {@link GuiBase} The GUI to draw on
	 * @param x           {@code int} Top left corner of tank
	 * @param y           {@code int} Top left corner of tank
	 * @param mouseX      {@code int} Mouse cursor position to check for tooltip
	 * @param mouseY      {@code int} Mouse cursor position to check for tooltip
	 * @param fluid       {@link FluidInstance} to draw in tank
	 * @param maxCapacity {@code int} Maximum tank capacity
	 * @param isTankEmpty {@code boolean} True if tank is empty
	 * @param layer       {@link GuiBase.Layer} The layer to draw on
	 */
	public void drawTank(class_332 drawContext, GuiBase<?> gui, int x, int y, int mouseX, int mouseY, FluidInstance fluid, FluidValue maxCapacity, boolean isTankEmpty, GuiBase.Layer layer) {
		if (gui.hideGuiElements()) return;
		if (layer == GuiBase.Layer.BACKGROUND) {
			x += gui.getGuiLeft();
			y += gui.getGuiTop();
		}

		int percentage = 0;
		FluidValue amount = FluidValue.EMPTY;
		if (!isTankEmpty) {
			amount = fluid.getAmount();
			percentage = percentage(maxCapacity.getRawValue(), amount.getRawValue());
		}
		drawSpriteStretched(drawContext, GuiSprites.TANK_BACKGROUND, x, y, 22, 56);
		if (!isTankEmpty) {
			drawFluid(drawContext, gui, fluid, x + 4, y + 4, 14, 48, maxCapacity.getRawValue());
		}
		drawSpriteStretched(drawContext, GuiSprites.TANK_FOREGROUND, x + 3, y + 3, 16, 50);

		if (gui.isPointInRect(x, y, 22, 56, mouseX, mouseY)) {
			List<class_2561> list = new ArrayList<>();
			if (isTankEmpty) {
				list.add(class_2561.method_43471("reborncore.gui.tooltip.tank_empty").method_27692(class_124.field_1065));
			} else {
				list.add(
						class_2561.method_43470(String.format("%s / %s", amount, maxCapacity))
								.method_27692(class_124.field_1065)
								.method_10852(SPACE_TEXT)
								.method_27693(FluidUtils.getFluidName(fluid))
				);
			}

			list.add(
					StringUtils.getPercentageText(percentage)
							.method_27692(class_124.field_1080)
							.method_10852(SPACE_TEXT)
							.method_10852(class_2561.method_43471("reborncore.gui.tooltip.tank_fullness"))
			);

			drawContext.method_51434(gui.method_64506(), list, mouseX, mouseY);
		}
	}

	/**
	 * Draws fluid in tank
	 *
	 * @param gui         {@link GuiBase} The GUI to draw on
	 * @param fluid       {@link FluidInstance} Fluid to draw
	 * @param x           {@code int} Top left corner of fluid
	 * @param y           {@code int} Top left corner of fluid
	 * @param width       {@code int} Width of fluid to draw
	 * @param height      {@code int} Height of fluid to draw
	 * @param maxCapacity {@code int} Maximum capacity of tank
	 */
	public void drawFluid(class_332 drawContext, GuiBase<?> gui, FluidInstance fluid, int x, int y, int width, int height, long maxCapacity) {
		if (fluid.fluid() == class_3612.field_15906) {
			return;
		}
		final class_1058 sprite = FluidVariantRendering.getSprite(fluid.fluidVariant());
		int color = FluidVariantRendering.getColor(fluid.fluidVariant());

		final int drawHeight = (int) (fluid.getAmount().getRawValue() / (maxCapacity * 1F) * height);
		y += height - drawHeight;
		int count = drawHeight / width;
		int remainder = drawHeight % width;
		for (int i = 0; i < count; i++) {
			drawContext.method_52710(class_10799.field_56883, sprite, x, y, width, width, color);
			y += width;
		}
		if (remainder != 0) {
			drawContext.method_44379(x, y, x + width, y + remainder);
			drawContext.method_52710(class_10799.field_56883, sprite, x, y, width, width, color);
			drawContext.method_44380();
		}
	}

	/**
	 * Draws burning progress, similar to vanilla furnace
	 *
	 * @param gui         {@link GuiBase} The GUI to draw on
	 * @param progress    {@code int} Current progress
	 * @param maxProgress {@code int} Maximum progress
	 * @param x           {@code int} Top left corner where to place burn bar
	 * @param y           {@code int} Top left corner where to place burn bar
	 * @param mouseX      {@code int} Mouse cursor position to check for tooltip
	 * @param mouseY      {@code int} Mouse cursor position to check for tooltip
	 * @param layer       {@link GuiBase.Layer} The layer to draw on
	 */
	public void drawBurnBar(class_332 drawContext, GuiBase<?> gui, int progress, int maxProgress, int x, int y, int mouseX, int mouseY, GuiBase.Layer layer) {
		if (gui.hideGuiElements()) return;
		if (layer == GuiBase.Layer.BACKGROUND) {
			x += gui.getGuiLeft();
			y += gui.getGuiTop();
		}
		drawContext.method_25290(class_10799.field_56883, GUI_ELEMENTS, x, y, 150, 64, 13, 13, 256, 256);
		int j = 13 - (int) ((double) progress / (double) maxProgress * 13);
		if (j > 0) {
			drawContext.method_25290(class_10799.field_56883, GUI_ELEMENTS, x, y + j, 150, 51 + j, 13, 13 - j, 256, 256);

		}
		if (gui.isPointInRect(x, y, 12, 12, mouseX, mouseY)) {
			int percentage = percentage(maxProgress, progress);
			List<class_2561> list = new ArrayList<>();
			list.add(StringUtils.getPercentageText(percentage));
			drawContext.method_51434(gui.method_64506(), list, mouseX, mouseY);
		}
	}

	/**
	 * Draws bar containing output slots
	 *
	 * @param x     {@code int} Top left corner where to place slots bar
	 * @param y     {@code int} Top left corner where to place slots bar
	 * @param count {@code int} Number of output slots
	 */
	public void drawOutputSlotBar(class_332 drawContext, int x, int y, int count) {
		drawSpriteStretched(drawContext, GuiSprites.SLOT_BAR_RIGHT, x, y, 3, 26);
		x += 3;
		for (int i = 1; i <= count; i++) {
			drawSpriteStretched(drawContext, GuiSprites.SLOT_BAR_CENTER, x, y, 20, 26);
			x += 20;
		}
		drawSpriteStretched(drawContext, GuiSprites.SLOT_BAR_LEFT, x, y, 3, 26);
	}

	protected int percentage(long MaxValue, long CurrentValue) {
		if (CurrentValue == 0) {
			return 0;
		}
		return (int) ((CurrentValue * 100.0f) / MaxValue);
	}

	public enum ProgressDirection {
		RIGHT(58, 150, 74, 150, 16, 10),
		LEFT(74, 160, 58, 160, 16, 10),
		DOWN(78, 170, 88, 170, 10, 16),
		UP(58, 170, 68, 170, 10, 16);
		public final class_4730 baseSprite;
		public final class_4730 overlaySprite;
		public final int x;
		public final int y;
		public final int xActive;
		public final int yActive;
		public final int width;
		public final int height;

		ProgressDirection(int x, int y, int xActive, int yActive, int width, int height) {
			this.baseSprite = GuiSprites.create("progress_%s_base".formatted(name().toLowerCase(Locale.ROOT)));
			this.overlaySprite = GuiSprites.create("progress_%s_overlay".formatted(name().toLowerCase(Locale.ROOT)));
			this.x = x;
			this.y = y;
			this.xActive = xActive;
			this.yActive = yActive;
			this.width = width;
			this.height = height;
		}
	}
}
