/*
 * This file is part of RebornCore, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2021 TeamReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.multiblock;

import reborncore.RebornCore;
import reborncore.api.blockentity.UnloadHandler;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2487;
import net.minecraft.class_2586;
import net.minecraft.class_2591;
import net.minecraft.class_2622;
import net.minecraft.class_2680;
import net.minecraft.class_3000;

/**
 * Base logic class for Multiblock-connected blockEntity entities. Most multiblock
 * machines should derive from this and implement their game logic in certain
 * abstract methods.
 */
public abstract class MultiblockBlockEntityBase extends IMultiblockPart implements class_3000, UnloadHandler {
	private MultiblockControllerBase controller;
	private boolean visited;

	private boolean saveMultiblockData;
	private class_2487 cachedMultiblockData;
	//private boolean paused;

	public MultiblockBlockEntityBase(class_2591<?> tBlockEntityType) {
		super(tBlockEntityType);
		controller = null;
		visited = false;
		saveMultiblockData = false;
		//paused = false;
		cachedMultiblockData = null;
	}

	// /// Multiblock Connection Base Logic
	@Override
	public Set<MultiblockControllerBase> attachToNeighbors() {
		Set<MultiblockControllerBase> controllers = null;
		MultiblockControllerBase bestController = null;

		// Look for a compatible controller in our neighboring parts.
		IMultiblockPart[] partsToCheck = getNeighboringParts();
		for (IMultiblockPart neighborPart : partsToCheck) {
			if (neighborPart.isConnected()) {
				MultiblockControllerBase candidate = neighborPart.getMultiblockController();
				if (!candidate.getClass().equals(this.getMultiblockControllerType())) {
					// Skip multiblocks with incompatible types
					continue;
				}

				if (controllers == null) {
					controllers = new HashSet<MultiblockControllerBase>();
					bestController = candidate;
				} else if (!controllers.contains(candidate) && candidate.shouldConsume(bestController)) {
					bestController = candidate;
				}

				controllers.add(candidate);
			}
		}

		// If we've located a valid neighboring controller, attach to it.
		if (bestController != null) {
			// attachBlock will call onAttached, which will set the controller.
			this.controller = bestController;
			bestController.attachBlock(this);
		}

		return controllers;
	}

	@Override
	public void assertDetached() {
		if (this.controller != null) {
			RebornCore.LOGGER.info(
				String.format("[assert] Part @ (%d, %d, %d) should be detached already, but detected that it was not. This is not a fatal error, and will be repaired, but is unusual.",
					method_11016().method_10263(), method_11016().method_10264(), method_11016().method_10260()));
			this.controller = null;
		}
	}

	// /// Overrides from base BlockEntity methods

	@Override
	public void method_11014(class_2680 blockState, class_2487 data) {
		super.method_11014(blockState, data);

		// We can't directly initialize a multiblock controller yet, so we cache
		// the data here until
		// we receive a validate() call, which creates the controller and hands
		// off the cached data.
		if (data.method_10545("multiblockData")) {
			this.cachedMultiblockData = data.method_10562("multiblockData");
		}
	}

	@Override
	public class_2487 method_11007(class_2487 data) {
		super.method_11007(data);

		if (isMultiblockSaveDelegate() && isConnected()) {
			class_2487 multiblockData = new class_2487();
			this.controller.write(multiblockData);
			data.method_10566("multiblockData", multiblockData);
		}
		return data;
	}

	@Override
	public void method_11012() {
		detachSelf(false);
		super.method_11012();
	}

	/**
	 * Called from Minecraft's blockEntity entity loop, after all blockEntity entities have
	 * been ticked, as the chunk in which this blockEntity entity is contained is
	 * unloading.
	 *
	 */
	@Override
	public void onUnload() {
		detachSelf(true);
	}

	/**
	 * This is called when a block is being marked as valid by the chunk, but
	 * has not yet fully been placed into the world's BlockEntity cache.
	 * this.worldObj, xCoord, yCoord and zCoord have been initialized, but any
	 * attempts to read data about the world can cause infinite loops - if you
	 * call getBlockEntity on this BlockEntity's coordinate from within
	 * validate(), you will blow your call stack.
	 * <p>
	 * TL;DR: Here there be dragons.
	 *
	 */
	@Override
	public void method_10996() {
		super.method_10996();
		MultiblockRegistry.onPartAdded(this.method_10997(), this);
	}

	// Network Communication
	@Override
	public class_2622 method_16886() {
		class_2487 packetData = new class_2487();
		encodeDescriptionPacket(packetData);
		return new class_2622(method_11016(), 0, packetData);
	}

	// /// Things to override in most implementations (IMultiblockPart)

	/**
	 * Override this to easily modify the description packet's data without
	 * having to worry about sending the packet itself. Decode this data in
	 * decodeDescriptionPacket.
	 *
	 * @param packetData An NBT compound tag into which you should write your custom
	 * description data.
	 */
	protected void encodeDescriptionPacket(class_2487 packetData) {
		if (this.isMultiblockSaveDelegate() && isConnected()) {
			class_2487 tag = new class_2487();
			getMultiblockController().formatDescriptionPacket(tag);
			packetData.method_10566("multiblockData", tag);
		}
	}

	/**
	 * Override this to easily read in data from a BlockEntity's description
	 * packet. Encoded in encodeDescriptionPacket.
	 *
	 * @param packetData The NBT data from the blockEntity entity's description packet.
	 */
	protected void decodeDescriptionPacket(class_2487 packetData) {
		if (packetData.method_10545("multiblockData")) {
			class_2487 tag = packetData.method_10562("multiblockData");
			if (isConnected()) {
				getMultiblockController().decodeDescriptionPacket(tag);
			} else {
				// This part hasn't been added to a machine yet, so cache the data.
				this.cachedMultiblockData = tag;
			}
		}
	}

	@Override
	public boolean hasMultiblockSaveData() {
		return this.cachedMultiblockData != null;
	}

	@Override
	public class_2487 getMultiblockSaveData() {
		return this.cachedMultiblockData;
	}

	@Override
	public void onMultiblockDataAssimilated() {
		this.cachedMultiblockData = null;
	}

	// /// Game logic callbacks (IMultiblockPart)

	@Override
	public abstract void onMachineAssembled(MultiblockControllerBase multiblockControllerBase);

	@Override
	public abstract void onMachineBroken();

	@Override
	public abstract void onMachineActivated();

	@Override
	public abstract void onMachineDeactivated();

	// /// Miscellaneous multiblock-assembly callbacks and support methods
	// (IMultiblockPart)

	@Override
	public boolean isConnected() {
		return (controller != null);
	}

	@Override
	public MultiblockControllerBase getMultiblockController() {
		return controller;
	}

	@Override
	public class_2338 getWorldLocation() {
		return this.method_11016();
	}

	@Override
	public void becomeMultiblockSaveDelegate() {
		this.saveMultiblockData = true;
	}

	@Override
	public void forfeitMultiblockSaveDelegate() {
		this.saveMultiblockData = false;
	}

	@Override
	public boolean isMultiblockSaveDelegate() {
		return this.saveMultiblockData;
	}

	@Override
	public void setUnvisited() {
		this.visited = false;
	}

	@Override
	public void setVisited() {
		this.visited = true;
	}

	@Override
	public boolean isVisited() {
		return this.visited;
	}

	@Override
	public void onAssimilated(MultiblockControllerBase newController) {
		assert (this.controller != newController);
		this.controller = newController;
	}

	@Override
	public void onAttached(MultiblockControllerBase newController) {
		this.controller = newController;
	}

	@Override
	public void onDetached(MultiblockControllerBase oldController) {
		this.controller = null;
	}

	@Override
	public abstract MultiblockControllerBase createNewMultiblock();

	@Override
	public IMultiblockPart[] getNeighboringParts() {
		class_2586 te;
		List<IMultiblockPart> neighborParts = new ArrayList<IMultiblockPart>();
		class_2338 neighborPosition, partPosition = this.getWorldLocation();

		for (class_2350 facing : class_2350.values()) {

			neighborPosition = partPosition.method_10093(facing);
			te = this.field_11863.method_8321(neighborPosition);

			if (te instanceof IMultiblockPart) {
				neighborParts.add((IMultiblockPart) te);
			}
		}

		return neighborParts.toArray(new IMultiblockPart[neighborParts.size()]);
	}

	@Override
	public void onOrphaned(MultiblockControllerBase controller, int oldSize, int newSize) {
		this.method_5431();
		method_10997().method_8524(method_11016(), this);
	}

	// // Helper functions for notifying neighboring blocks
	protected void notifyNeighborsOfBlockChange() {
		field_11863.method_8452(method_11016(), method_11010().method_26204());
	}

	protected void notifyNeighborsOfBlockEntityChange() {
		field_11863.method_8452(method_11016(), method_11010().method_26204());
	}

	// /// Private/Protected Logic Helpers
	/*
	 * Detaches this block from its controller. Calls detachBlock() and clears
	 * the controller member.
	 */
	protected void detachSelf(boolean chunkUnloading) {
		if (this.controller != null) {
			// Clean part out of controller
			this.controller.detachBlock(this, chunkUnloading);

			// The above should call onDetached, but, just in case...
			this.controller = null;
		}

		// Clean part out of lists in the registry
		MultiblockRegistry.onPartRemovedFromWorld(method_10997(), this);
	}

	@Override
	public class_2680 method_11010() {
		return field_11863.method_8320(field_11867);
	}
}
