/*
 * This file is part of RebornCore, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2021 TeamReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.crafting.ingredient;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonSyntaxException;
import com.mojang.serialization.Dynamic;
import com.mojang.serialization.JsonOps;
import org.apache.commons.lang3.Validate;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1856;
import net.minecraft.class_2378;
import net.minecraft.class_2487;
import net.minecraft.class_2509;
import net.minecraft.class_2960;
import net.minecraft.class_3518;

public class StackIngredient extends RebornIngredient {

	private final List<class_1799> stacks;

	private final Optional<Integer> count;
	private final Optional<class_2487> tag;
	private final boolean requireEmptyTag;

	public StackIngredient(List<class_1799> stacks, Optional<Integer> count, Optional<class_2487> tag, boolean requireEmptyTag) {
		super(IngredientManager.STACK_RECIPE_TYPE);
		this.stacks = stacks;
		this.count = count;
		this.tag = tag;
		this.requireEmptyTag = requireEmptyTag;
		Validate.isTrue(stacks.size() == 1, "stack size must 1");
	}

	public static RebornIngredient deserialize(JsonObject json) {
		if (!json.has("item")) {
			System.out.println("nope");
		}
		class_2960 identifier = new class_2960(class_3518.method_15265(json, "item"));
		class_1792 item = class_2378.field_11142.method_17966(identifier).orElseThrow(() -> new JsonSyntaxException("Unknown item '" + identifier + "'"));

		Optional<Integer> stackSize = Optional.empty();
		if (json.has("count")) {
			stackSize = Optional.of(class_3518.method_15260(json, "count"));
		}

		Optional<class_2487> tag = Optional.empty();
		boolean requireEmptyTag = false;

		if (json.has("nbt")) {
			if (!json.get("nbt").isJsonObject()) {
				if (json.get("nbt").getAsString().equals("empty")) {
					requireEmptyTag = true;
				}
			} else {
				tag = Optional.of((class_2487) Dynamic.convert(JsonOps.INSTANCE, class_2509.field_11560, json.get("nbt")));
			}
		}

		return new StackIngredient(Collections.singletonList(new class_1799(item)), stackSize, tag, requireEmptyTag);
	}


	@Override
	public boolean test(class_1799 itemStack) {
		if (itemStack.method_7960()) {
			return false;
		}
		if (stacks.stream().noneMatch(recipeStack -> recipeStack.method_7909() == itemStack.method_7909())) {
			return false;
		}
		if (count.isPresent() && count.get() > itemStack.method_7947()) {
			return false;
		}
		if (tag.isPresent()) {
			if (!itemStack.method_7985()) {
				return false;
			}

			//Bit of a meme here, as DataFixer likes to use the most basic primative type over using an int.
			//So we have to go to json and back on the incoming stack to be sure its using types that match our input.

			class_2487 compoundTag = itemStack.method_7969();
			JsonElement jsonElement = Dynamic.convert(class_2509.field_11560, JsonOps.INSTANCE, compoundTag);
			compoundTag = (class_2487) Dynamic.convert(JsonOps.INSTANCE, class_2509.field_11560, jsonElement);

			if (!tag.get().equals(compoundTag)) {
				return false;
			}
		}
		return !requireEmptyTag || !itemStack.method_7985();
	}

	@Override
	public class_1856 getPreview() {
		return class_1856.method_26964(getPreviewStacks().stream());
	}

	@Override
	public List<class_1799> getPreviewStacks() {
		return Collections.unmodifiableList(
				stacks.stream()
						.map(class_1799::method_7972)
						.peek(itemStack -> itemStack.method_7939(count.orElse(1)))
						.peek(itemStack -> itemStack.method_7980(tag.orElse(null)))
						.collect(Collectors.toList()));
	}

	@Override
	public JsonObject toJson() {
		JsonObject jsonObject = new JsonObject();

		jsonObject.addProperty("item", class_2378.field_11142.method_10221(stacks.get(0).method_7909()).toString());
		count.ifPresent(integer -> jsonObject.addProperty("count", integer));

		if (requireEmptyTag) {
			jsonObject.addProperty("nbt", "empty");
		} else {
			tag.ifPresent(compoundTag -> jsonObject.add("nbt", Dynamic.convert(class_2509.field_11560, JsonOps.INSTANCE, compoundTag)));
		}

		return jsonObject;
	}

	@Override
	public int getCount() {
		return count.orElse(1);
	}
}
