/*
 * This file is part of RebornCore, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2021 TeamReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.chunkloading;

import com.mojang.serialization.Codec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import reborncore.common.network.ClientBoundPackets;
import reborncore.common.network.NetworkManager;

import java.util.*;
import java.util.stream.Collectors;
import net.minecraft.class_18;
import net.minecraft.class_1923;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2487;
import net.minecraft.class_2509;
import net.minecraft.class_2960;
import net.minecraft.class_3215;
import net.minecraft.class_3218;
import net.minecraft.class_3222;
import net.minecraft.class_3230;
import net.minecraft.class_5321;

//This does not do the actual chunk loading, just keeps track of what chunks the chunk loader has loaded
public class ChunkLoaderManager extends class_18 {

	public static Codec<List<LoadedChunk>> CODEC = Codec.list(LoadedChunk.CODEC);

	private static final class_3230<class_1923> CHUNK_LOADER = class_3230.method_14291("reborncore:chunk_loader", Comparator.comparingLong(class_1923::method_8324));
	private static final String KEY = "reborncore_chunk_loader";

	public ChunkLoaderManager() {
		super(KEY);
	}

	public static ChunkLoaderManager get(class_1937 world){
		class_3218 serverWorld = (class_3218) world;
		return serverWorld.method_17983().method_17924(ChunkLoaderManager::new, KEY);
	}

	private final List<LoadedChunk> loadedChunks = new ArrayList<>();

	@Override
	public void method_77(class_2487 tag) {
		loadedChunks.clear();

		List<LoadedChunk> chunks = CODEC.parse(class_2509.field_11560, tag.method_10562("loadedchunks"))
				.result()
				.orElse(Collections.emptyList());

		loadedChunks.addAll(chunks);
	}

	@Override
	public class_2487 method_75(class_2487 compoundTag) {
		CODEC.encodeStart(class_2509.field_11560, loadedChunks)
				.result()
				.ifPresent(tag -> compoundTag.method_10566("loadedchunks", tag));
		return compoundTag;
	}

	public Optional<LoadedChunk> getLoadedChunk(class_1937 world, class_1923 chunkPos, class_2338 chunkLoader){
		return loadedChunks.stream()
			.filter(loadedChunk -> loadedChunk.getWorld().equals(getWorldName(world)))
			.filter(loadedChunk -> loadedChunk.getChunk().equals(chunkPos))
			.filter(loadedChunk -> loadedChunk.getChunkLoader().equals(chunkLoader))
			.findFirst();
	}

	public Optional<LoadedChunk> getLoadedChunk(class_1937 world, class_1923 chunkPos){
		return loadedChunks.stream()
			.filter(loadedChunk -> loadedChunk.getWorld().equals(getWorldName(world)))
			.filter(loadedChunk -> loadedChunk.getChunk().equals(chunkPos))
			.findFirst();
	}

	public List<LoadedChunk> getLoadedChunks(class_1937 world, class_2338 chunkloader){
		return loadedChunks.stream()
			.filter(loadedChunk -> loadedChunk.getWorld().equals(getWorldName(world)))
			.filter(loadedChunk -> loadedChunk.getChunkLoader().equals(chunkloader))
			.collect(Collectors.toList());
	}

	public boolean isChunkLoaded(class_1937 world, class_1923 chunkPos, class_2338 chunkLoader){
		return getLoadedChunk(world, chunkPos, chunkLoader).isPresent();
	}

	public boolean isChunkLoaded(class_1937 world, class_1923 chunkPos){
		return getLoadedChunk(world, chunkPos).isPresent();
	}


	public void loadChunk(class_1937 world, class_1923 chunkPos, class_2338 chunkLoader, String player){
		Validate.isTrue(!isChunkLoaded(world, chunkPos, chunkLoader), "chunk is already loaded");
		LoadedChunk loadedChunk = new LoadedChunk(chunkPos, getWorldName(world), player, chunkLoader);
		loadedChunks.add(loadedChunk);

		final class_3215 serverChunkManager = ((class_3218) world).method_14178();
		serverChunkManager.method_17297(ChunkLoaderManager.CHUNK_LOADER, loadedChunk.getChunk(), 31, loadedChunk.getChunk());

		method_80();
	}

	public void unloadChunkLoader(class_1937 world, class_2338 chunkLoader){
		getLoadedChunks(world, chunkLoader).forEach(loadedChunk -> unloadChunk(world, loadedChunk.getChunk(), chunkLoader));
	}

	public void unloadChunk(class_1937 world, class_1923 chunkPos, class_2338 chunkLoader){
		Optional<LoadedChunk> optionalLoadedChunk = getLoadedChunk(world, chunkPos, chunkLoader);
		Validate.isTrue(optionalLoadedChunk.isPresent(), "chunk is not loaded");

		LoadedChunk loadedChunk = optionalLoadedChunk.get();

		loadedChunks.remove(loadedChunk);

		if(!isChunkLoaded(world, loadedChunk.getChunk())){
			final class_3215 serverChunkManager = ((class_3218) world).method_14178();
			serverChunkManager.method_17300(ChunkLoaderManager.CHUNK_LOADER, loadedChunk.getChunk(), 31, loadedChunk.getChunk());
		}
		method_80();
	}

	public static class_2960 getWorldName(class_1937 world){
		return world.method_27983().method_29177();
	}

	public static class_5321<class_1937> getDimensionRegistryKey(class_1937 world){
		return world.method_27983();
	}

	public void syncChunkLoaderToClient(class_3222 serverPlayerEntity, class_2338 chunkLoader){
		syncToClient(serverPlayerEntity, loadedChunks.stream().filter(loadedChunk -> loadedChunk.getChunkLoader().equals(chunkLoader)).collect(Collectors.toList()));
	}

	public void syncAllToClient(class_3222 serverPlayerEntity) {
		syncToClient(serverPlayerEntity, loadedChunks);
	}

	public void clearClient(class_3222 serverPlayerEntity) {
		syncToClient(serverPlayerEntity, Collections.emptyList());
	}

	public void syncToClient(class_3222 serverPlayerEntity, List<LoadedChunk> chunks) {
		NetworkManager.sendToPlayer(ClientBoundPackets.createPacketSyncLoadedChunks(chunks), serverPlayerEntity);
	}

	public static class LoadedChunk {

		public static Codec<class_1923> CHUNK_POS_CODEC = RecordCodecBuilder.create(instance ->
				instance.group(
					Codec.INT.fieldOf("x").forGetter(p -> p.field_9181),
					Codec.INT.fieldOf("z").forGetter(p -> p.field_9180)
				)
				.apply(instance, class_1923::new));

		public static Codec<LoadedChunk> CODEC = RecordCodecBuilder.create(instance ->
				instance.group(
					CHUNK_POS_CODEC.fieldOf("chunk").forGetter(LoadedChunk::getChunk),
					class_2960.field_25139.fieldOf("world").forGetter(LoadedChunk::getWorld),
					Codec.STRING.fieldOf("player").forGetter(LoadedChunk::getPlayer),
					class_2338.field_25064.fieldOf("chunkLoader").forGetter(LoadedChunk::getChunkLoader)
				)
				.apply(instance, LoadedChunk::new));

		private class_1923 chunk;
		private class_2960 world;
		private String player;
		private class_2338 chunkLoader;

		public LoadedChunk(class_1923 chunk, class_2960 world, String player, class_2338 chunkLoader) {
			this.chunk = chunk;
			this.world = world;
			this.player = player;
			this.chunkLoader = chunkLoader;
			Validate.isTrue(!StringUtils.isBlank(player), "Player cannot be blank");
		}

		public class_1923 getChunk() {
			return chunk;
		}

		public class_2960 getWorld() {
			return world;
		}

		public String getPlayer() {
			return player;
		}

		public class_2338 getChunkLoader() {
			return chunkLoader;
		}
	}
}
