/*
 * This file is part of RebornCore, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2021 TeamReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.blockentity;

import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.fabricmc.fabric.api.blockrenderlayer.v1.BlockRenderLayerMap;
import net.minecraft.class_1799;
import net.minecraft.class_1920;
import net.minecraft.class_1922;
import net.minecraft.class_2246;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2404;
import net.minecraft.class_2680;
import net.minecraft.class_310;
import net.minecraft.class_3610;
import net.minecraft.class_4587;
import net.minecraft.class_4588;
import net.minecraft.class_4597;
import net.minecraft.class_4608;
import net.minecraft.class_4696;
import net.minecraft.class_776;
import net.minecraft.class_809;
import net.minecraft.client.render.*;
import java.util.Random;
import java.util.function.BiPredicate;

/**
 * Writes a multiblock for either verification or hologram rendering
 *
 * @see MultiblockVerifier
 * @see HologramRenderer
 * @author ramidzkh
 */
public interface MultiblockWriter {

	/**
	 * Adds a block to the multiblock
	 *
	 * @param x X
	 * @param y Y
	 * @param z Z
	 * @param predicate Predicate of the position
	 * @param state The state for the hologram
	 * @return This. Useful for chaining
	 */
	MultiblockWriter add(int x, int y, int z, BiPredicate<class_1922, class_2338> predicate, class_2680 state);

	/**
	 * Fills a section between (ax, ay, az) to (bx, by, bz)
	 *
	 * @param ax X of the first point
	 * @param ay Y of the first point
	 * @param az Z of the first point
	 * @param bx X of the second point
	 * @param by X of the second point
	 * @param bz Z of the second point
	 * @param predicate Predicate of the position
	 * @param state The state for the hologram
	 * @return This. Useful for chaining
	 */
	default MultiblockWriter fill(int ax, int ay, int az, int bx, int by, int bz, BiPredicate<class_1922, class_2338> predicate, class_2680 state) {
		for (int x = ax; x < bx; x++) {
			for (int y = ay; y < by; y++) {
				for (int z = az; z < bz; z++) {
					add(x, y, z, predicate, state);
				}
			}
		}

		return this;
	}

	/**
	 * Fills the outer ring of (0, 0, 0) to (pX, pY, pZ) through the axis, using the <code>predicate</code> and
	 * <code>state</code>. The inside of the ring uses <code>holePredicate</code> and <code>holeHologramState</code>
	 *
	 * @param through The axis to go through
	 * @param pX Size on the X axis
	 * @param pY Size on the Y axis
	 * @param pZ Size on the Z axis
	 * @param predicate Predicate for the ring
	 * @param state The ring state for the hologram
	 * @param holePredicate Predicate for the hole
	 * @param holeHologramState The hole state for the hologram
	 * @return This. Useful for chaining
	 */
	default MultiblockWriter ring(class_2350.class_2351 through, int pX, int pY, int pZ, BiPredicate<class_1922, class_2338> predicate, class_2680 state, BiPredicate<class_1922, class_2338> holePredicate, class_2680 holeHologramState) {
		if (holePredicate == null) {
			holePredicate = predicate.negate();
		}

		if (holeHologramState == null) {
			holeHologramState = class_2246.field_10124.method_9564();
		}

		if (through == class_2350.class_2351.field_11048) {
			for (int y = 0; y < pY; y++) {
				for (int z = 0; z < pZ; z++) {
					if ((y == 0 || y == (pY - 1)) || (z == 0 || z == (pZ - 1))) {
						add(pX, y, z, predicate, state);
					} else {
						add(pX, y, z, holePredicate, holeHologramState);
					}
				}
			}
		} else if (through == class_2350.class_2351.field_11052) {
			for (int x = 0; x < pX; x++) {
				for (int z = 0; z < pZ; z++) {
					if ((x == 0 || x == (pX - 1)) || (z == 0 || z == (pZ - 1))) {
						add(x, pY, z, predicate, state);
					} else {
						add(x, pY, z, holePredicate, holeHologramState);
					}
				}
			}
		} else if (through == class_2350.class_2351.field_11051) {
			for (int x = 0; x < pX; x++) {
				for (int y = 0; y < pY; y++) {
					if ((x == 0 || x == (pX - 1)) || (y == 0 || y == (pY - 1))) {
						add(x, y, pZ, predicate, state);
					} else {
						add(x, y, pZ, holePredicate, holeHologramState);
					}
				}
			}
		}

		return this;
	}

	default MultiblockWriter ringWithAir(class_2350.class_2351 through, int x, int y, int z, BiPredicate<class_1922, class_2338> predicate, class_2680 state) {
		return ring(through, x, y, z, predicate, state, (view, pos) -> view.method_8320(pos).method_26204() == class_2246.field_10124, class_2246.field_10124.method_9564());
	}

	default MultiblockWriter add(int x, int y, int z, class_2680 state) {
		return this.add(x, y, z, (view, pos) -> view.method_8320(pos) == state, state);
	}

	default MultiblockWriter fill(int ax, int ay, int az, int bx, int by, int bz, class_2680 state) {
		return fill(ax, ay, az, bx, by, bz, (view, pos) -> view.method_8320(pos) == state, state);
	}

	default MultiblockWriter ring(class_2350.class_2351 through, int x, int y, int z, class_2680 state, class_2680 holeState) {
		return ring(through, x, y, z, (view, pos) -> view.method_8320(pos) == state, state, (view, pos) -> view.method_8320(pos) == holeState, holeState);
	}

	default MultiblockWriter ringWithAir(class_2350.class_2351 through, int x, int y, int z, class_2680 state) {
		return ringWithAir(through, x, y, z, (view, pos) -> view.method_8320(pos) == state, state);
	}

	default MultiblockWriter translate(int offsetX, int offsetY, int offsetZ) {
		return (x, y, z, predicate, state) -> add(offsetX + x, offsetY + y, offsetZ + z, predicate, state);
	}

	default MultiblockWriter rotate() {
		return (x, y, z, predicate, state) -> add(-z, y, x, predicate, state);
	}

	default MultiblockWriter rotate(class_2350 direction) {
		MultiblockWriter w = this;

		switch (direction) {
			case field_11043:
				w = w.rotate();
			case field_11039:
				w = w.rotate();
			case field_11035:
				w = w.rotate();
		}

		return w;
	}

	/**
	 * A writer which prints the hologram to {@link System#out}
	 */
	class DebugWriter implements MultiblockWriter {
		private final MultiblockWriter writer;

		public DebugWriter(MultiblockWriter writer) {
			this.writer = writer;
		}

		@Override
		public MultiblockWriter add(int x, int y, int z, BiPredicate<class_1922, class_2338> predicate, class_2680 state) {
			System.out.printf("\t%d\t%d\t%d\t%s\n", x, y, z, state.method_26204());

			if (writer != null) {
				writer.add(x, y, z, predicate, state);
			}

			return this;
		}
	}

	/**
	 * A writer which verifies the positions of each block
	 */
	class MultiblockVerifier implements MultiblockWriter {
		private final class_2338 relative;
		private final class_1922 view;

		private boolean valid = true;

		public MultiblockVerifier(class_2338 relative, class_1922 view) {
			this.relative = relative;
			this.view = view;
		}

		public boolean isValid() {
			return valid;
		}

		@Override
		public MultiblockWriter add(int x, int y, int z, BiPredicate<class_1922, class_2338> predicate, class_2680 state) {
			if (valid) {
				valid = predicate.test(view, relative.method_10069(x, y, z));
			}

			return this;
		}
	}

	/**
	 * Renders a hologram
	 */
	@Environment(EnvType.CLIENT)
	class HologramRenderer implements MultiblockWriter {
		private static final class_2338 OUT_OF_WORLD_POS = new class_2338(0, 260, 0); // Bad hack; disables lighting

		private final class_1920 view;
		private final class_4587 matrix;
		private final class_4597 vertexConsumerProvider;
		private final float scale;

		public HologramRenderer(class_1920 view, class_4587 matrix, class_4597 vertexConsumerProvider, float scale) {
			this.view = view;
			this.matrix = matrix;
			this.vertexConsumerProvider = vertexConsumerProvider;
			this.scale = scale;
		}

		@Override
		public MultiblockWriter add(int x, int y, int z, BiPredicate<class_1922, class_2338> predicate, class_2680 state) {
			final class_776 blockRenderManager = class_310.method_1551().method_1541();
			matrix.method_22903();
			matrix.method_22904(x, y, z);
			matrix.method_22904(0.5, 0.5, 0.5);
			matrix.method_22905(scale, scale, scale);


			if (state.method_26204() instanceof class_2404) {
				class_3610 fluidState = ((class_2404) state.method_26204()).method_9545(state);
				class_310.method_1551().method_1480().method_23178(new class_1799(fluidState.method_15772().method_15774()), class_809.class_811.field_4319, 15728880, class_4608.field_21444, matrix, vertexConsumerProvider);
			} else {
				matrix.method_22904(-0.5, -0.5, -0.5);
				class_4588 consumer = vertexConsumerProvider.getBuffer(class_4696.method_23679(state));
				blockRenderManager.method_3355(state, OUT_OF_WORLD_POS, view, matrix, consumer, false, new Random());
			}
			
			matrix.method_22909();
			return this;
		}
	}
}
