/*
 * This file is part of RebornCore, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2021 TeamReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.client.gui.builder.slot;

import com.google.common.collect.Lists;
import com.mojang.blaze3d.systems.RenderSystem;
import reborncore.client.gui.GuiUtil;
import reborncore.client.gui.builder.GuiBase;
import reborncore.client.gui.builder.slot.elements.ConfigSlotElement;
import reborncore.client.gui.builder.slot.elements.ElementBase;
import reborncore.client.gui.builder.slot.elements.SlotType;
import reborncore.client.screen.builder.BuiltScreenHandler;
import reborncore.common.blockentity.MachineBaseBlockEntity;
import reborncore.common.network.NetworkManager;
import reborncore.common.network.ServerBoundPackets;
import reborncore.common.util.Color;
import reborncore.mixin.common.AccessorSlot;

import org.jetbrains.annotations.Nullable;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.stream.Collectors;
import net.minecraft.class_156;
import net.minecraft.class_1735;
import net.minecraft.class_2585;
import net.minecraft.class_310;
import net.minecraft.class_4587;

public class SlotConfigGui {

	static HashMap<Integer, ConfigSlotElement> slotElementMap = new HashMap<>();

	public static int selectedSlot = 0;

	public static void reset() {
		selectedSlot = -1;
	}

	public static void init(GuiBase<?> guiBase) {
		reset();
		slotElementMap.clear();

		BuiltScreenHandler container = guiBase.builtScreenHandler;
		for (class_1735 slot : container.field_7761) {
			if (guiBase.be != slot.field_7871) {
				continue;
			}
			AccessorSlot accessorSlot = (AccessorSlot) slot;
			ConfigSlotElement slotElement = new ConfigSlotElement(guiBase.getMachine().getOptionalInventory().get(), accessorSlot.getIndex(), SlotType.NORMAL, slot.field_7873 - guiBase.getGuiLeft() + 50, slot.field_7872 - guiBase.getGuiTop() - 25, guiBase);
			slotElementMap.put(accessorSlot.getIndex(), slotElement);
		}

	}

	public static void draw(class_4587 matrixStack, GuiBase<?> guiBase, int mouseX, int mouseY) {
		BuiltScreenHandler container = guiBase.builtScreenHandler;
		for (class_1735 slot : container.field_7761) {
			if (guiBase.be != slot.field_7871) {
				continue;
			}
			RenderSystem.color3f(255, 0, 0);
			Color color = new Color(255, 0, 0, 128);
			GuiUtil.drawGradientRect(slot.field_7873 - 1, slot.field_7872 - 1, 18, 18, color.getColor(), color.getColor());
			RenderSystem.color3f(255, 255, 255);
		}

		if (selectedSlot != -1) {

			slotElementMap.get(selectedSlot).draw(matrixStack, guiBase);
		}
	}

	public static List<ConfigSlotElement> getVisibleElements() {
		if (selectedSlot == -1) {
			return Collections.emptyList();
		}
		return slotElementMap.values().stream()
				.filter(configSlotElement -> configSlotElement.getId() == selectedSlot)
				.collect(Collectors.toList());
	}

	public static void copyToClipboard() {
		MachineBaseBlockEntity machine = getMachine();
		if (machine == null || machine.getSlotConfiguration() == null) {
			return;
		}
		String json = machine.getSlotConfiguration().toJson(machine.getClass().getCanonicalName());
		class_310.method_1551().field_1774.method_1455(json);
		class_310.method_1551().field_1724.method_9203(new class_2585("Slot configuration copyied to clipboard"), class_156.field_25140);
	}

	public static void pasteFromClipboard() {
		MachineBaseBlockEntity machine = getMachine();
		if (machine == null || machine.getSlotConfiguration() == null) {
			return;
		}
		String json = class_310.method_1551().field_1774.method_1460();
		try {
			machine.getSlotConfiguration().readJson(json, machine.getClass().getCanonicalName());
			NetworkManager.sendToServer(ServerBoundPackets.createPacketConfigSave(machine.method_11016(), machine.getSlotConfiguration()));
			class_310.method_1551().field_1724.method_9203(new class_2585("Slot configuration loaded from clipboard"), class_156.field_25140);
		} catch (UnsupportedOperationException e) {
			class_310.method_1551().field_1724.method_9203(new class_2585(e.getMessage()), class_156.field_25140);
		}
	}

	@Nullable
	private static MachineBaseBlockEntity getMachine() {
		if (!(class_310.method_1551().field_1755 instanceof GuiBase)) {
			return null;
		}
		GuiBase<?> base = (GuiBase<?>) class_310.method_1551().field_1755;
		if (!(base.be instanceof MachineBaseBlockEntity)) {
			return null;
		}
		MachineBaseBlockEntity machineBase = (MachineBaseBlockEntity) base.be;
		return machineBase;
	}

	public static boolean mouseClicked(GuiBase<?> guiBase, double mouseX, double mouseY, int mouseButton) {
		if (mouseButton == 0) {
			for (ConfigSlotElement configSlotElement : getVisibleElements()) {
				for (ElementBase element : configSlotElement.elements) {
					if (element.isInRect(guiBase, element.x, element.y, element.getWidth(guiBase.getMachine()), element.getHeight(guiBase.getMachine()), mouseX, mouseY)) {
						element.isPressing = true;
						boolean action = element.onStartPress(guiBase.getMachine(), guiBase, mouseX, mouseY);
						for (ElementBase e : getVisibleElements()) {
							if (e != element) {
								e.isPressing = false;
							}
						}
						if (action) {
							break;
						}
					} else {
						element.isPressing = false;
					}
				}
			}
		}
		BuiltScreenHandler screenHandler = guiBase.builtScreenHandler;

		if (getVisibleElements().isEmpty()) {
			for (class_1735 slot : screenHandler.field_7761) {
				if (guiBase.be != slot.field_7871) {
					continue;
				}
				if (guiBase.isPointInRect(slot.field_7873, slot.field_7872, 18, 18, mouseX, mouseY)) {
					AccessorSlot accessorSlot = (AccessorSlot) slot;
					selectedSlot = accessorSlot.getIndex();
					return true;
				}
			}
		}
		return !getVisibleElements().isEmpty();
	}

	public static void mouseClickMove(double mouseX, double mouseY, int mouseButton, long timeSinceLastClick, GuiBase<?> guiBase) {
		if (mouseButton == 0) {
			for (ConfigSlotElement configSlotElement : getVisibleElements()) {
				for (ElementBase element : configSlotElement.elements) {
					if (element.isInRect(guiBase, element.x, element.y, element.getWidth(guiBase.getMachine()), element.getHeight(guiBase.getMachine()), mouseX, mouseY)) {
						element.isDragging = true;
						boolean action = element.onDrag(guiBase.getMachine(), guiBase, mouseX, mouseY);
						for (ElementBase e : getVisibleElements()) {
							if (e != element) {
								e.isDragging = false;
							}
						}
						if (action) {
							break;
						}
					} else {
						element.isDragging = false;
					}
				}
			}
		}
	}

	public static boolean mouseReleased(GuiBase<?> guiBase, double mouseX, double mouseY, int mouseButton) {
		boolean clicked = false;
		if (mouseButton == 0) {
			for (ConfigSlotElement configSlotElement : getVisibleElements()) {
				if (configSlotElement.isInRect(guiBase, configSlotElement.x, configSlotElement.y, configSlotElement.getWidth(guiBase.getMachine()), configSlotElement.getHeight(guiBase.getMachine()), mouseX, mouseY)) {
					clicked = true;
				}
				for (ElementBase element : Lists.reverse(configSlotElement.elements)) {
					if (element.isInRect(guiBase, element.x, element.y, element.getWidth(guiBase.getMachine()), element.getHeight(guiBase.getMachine()), mouseX, mouseY)) {
						element.isReleasing = true;
						boolean action = element.onRelease(guiBase.getMachine(), guiBase, mouseX, mouseY);
						for (ElementBase e : getVisibleElements()) {
							if (e != element) {
								e.isReleasing = false;
							}
						}
						if (action) {
							clicked = true;
						}
						break;
					} else {
						element.isReleasing = false;
					}
				}
			}
		}
		return clicked;
	}

}
