/*
 * This file is part of RebornCore, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2021 TeamReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.util;

import reborncore.common.recipes.IRecipeInput;
import team.reborn.energy.Energy;

import java.util.List;
import java.util.function.Predicate;
import net.minecraft.class_124;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_2487;
import net.minecraft.class_2561;
import net.minecraft.class_2588;

/**
 * Created by mark on 12/04/15.
 */
public class ItemUtils {

	public static boolean isItemEqual(final class_1799 a, final class_1799 b,
									  final boolean matchNBT) {
		if (a.method_7960() || b.method_7960()) {
			return false;
		}
		if (a.method_7909() != b.method_7909()) {
			return false;
		}
		return !matchNBT || class_1799.method_7975(a, b);
	}

	public static boolean isItemEqual(class_1799 a, class_1799 b, boolean matchNBT,
									  boolean useTags) {
		if (a.method_7960() && b.method_7960()) {
			return true;
		}
		if (isItemEqual(a, b, matchNBT)) {
			return true;
		}
		if (a.method_7960() || b.method_7960()) {
			return false;
		}
		if (useTags) {

			//TODO tags
		}
		return false;
	}

	//TODO tags
	public static boolean isInputEqual(Object input, class_1799 other, boolean matchNBT,
									   boolean useTags) {
		if (input instanceof class_1799) {
			return isItemEqual((class_1799) input, other, matchNBT, useTags);
		} else if (input instanceof String) {
			//TODO tags
		} else if (input instanceof IRecipeInput) {
			List<class_1799> inputs = ((IRecipeInput) input).getAllStacks();
			for (class_1799 stack : inputs) {
				if (isItemEqual(stack, other, matchNBT, false)) {
					return true;
				}
			}
		}
		return false;
	}

	public static void writeItemToNBT(class_1799 stack, class_2487 data) {
		if (stack.method_7960() || stack.method_7947() <= 0) {
			return;
		}
		if (stack.method_7947() > 127) {
			stack.method_7939(127);
		}
		stack.method_7953(data);
	}

	public static class_1799 readItemFromNBT(class_2487 data) {
		return class_1799.method_7915(data);
	}

	public static double getPowerForDurabilityBar(class_1799 stack) {
		if (stack.method_7960()) {
			return 0.0;
		}

		if (!Energy.valid(stack)) {
			return 0.0;
		}

		return Energy.of(stack).getEnergy() / Energy.of(stack).getMaxStored();
	}

	/**
	 * Checks if powered item is active
	 *
	 * @param stack ItemStack ItemStack to check
	 * @return True if powered item is active
	 */
	public static boolean isActive(class_1799 stack) {
		return !stack.method_7960() && stack.method_7969() != null && stack.method_7969().method_10577("isActive");
	}

	/**
	 * Check if powered item has enough energy to continue being in active state
	 *
	 * @param stack     ItemStack ItemStack to check
	 * @param cost      int Cost of operation performed by tool
	 * @param isClient  boolean Client side
	 * @param messageId int MessageID for sending no spam message
	 */
	public static void checkActive(class_1799 stack, int cost, boolean isClient, int messageId) {
		if (!ItemUtils.isActive(stack)) {
			return;
		}
		if (Energy.of(stack).getEnergy() >= cost) {
			return;
		}
		if (isClient) {
			ChatUtils.sendNoSpamMessages(messageId, new class_2588("reborncore.message.energyError")
					.method_27692(class_124.field_1080)
					.method_27693(" ")
					.method_10852(
							new class_2588("reborncore.message.deactivating")
									.method_27692(class_124.field_1065)
					)
			);
		}
		stack.method_7948().method_10556("isActive", false);
	}

	/**
	 * Switch active\inactive state for powered item
	 *
	 * @param stack     ItemStack ItemStack to work on
	 * @param cost      int Cost of operation performed by tool
	 * @param isClient  boolean Are we on client side
	 * @param messageId MessageID for sending no spam message
	 */
	public static void switchActive(class_1799 stack, int cost, boolean isClient, int messageId) {
		ItemUtils.checkActive(stack, cost, isClient, messageId);

		if (!ItemUtils.isActive(stack)) {
			stack.method_7948().method_10556("isActive", true);
			if (isClient) {


				ChatUtils.sendNoSpamMessages(messageId, new class_2588("reborncore.message.setTo")
						.method_27692(class_124.field_1080)
						.method_27693(" ")
						.method_10852(
								new class_2588("reborncore.message.active")
										.method_27692(class_124.field_1065)
						)
				);
			}
		} else {
			stack.method_7948().method_10556("isActive", false);
			if (isClient) {
				ChatUtils.sendNoSpamMessages(messageId, new class_2588("reborncore.message.setTo")
						.method_27692(class_124.field_1080)
						.method_27693(" ")
						.method_10852(
								new class_2588("reborncore.message.inactive")
										.method_27692(class_124.field_1065)
						)
				);
			}
		}
	}

	/**
	 * Adds active\inactive state to powered item tooltip
	 *
	 * @param stack   ItemStack ItemStack to check
	 * @param tooltip List Tooltip strings
	 */
	public static void buildActiveTooltip(class_1799 stack, List<class_2561> tooltip) {
		if (!ItemUtils.isActive(stack)) {
			tooltip.add(new class_2588("reborncore.message.inactive").method_27692(class_124.field_1061));
		} else {
			tooltip.add(new class_2588("reborncore.message.active").method_27692(class_124.field_1060));
		}
	}

	/**
	 *  Output energy from item to other items in inventory
	 *
	 * @param player PlayerEntity having powered item
	 * @param itemStack ItemStack Powered item
	 * @param maxOutput int Maximum output rate of powered item
	 */
	public static void distributePowerToInventory(class_1657 player, class_1799 itemStack, int maxOutput) {
		distributePowerToInventory(player, itemStack, maxOutput, (stack) -> true);
	}

	public static void distributePowerToInventory(class_1657 player, class_1799 itemStack, int maxOutput, Predicate<class_1799> filter) {
		if (!Energy.valid(itemStack)) {
			return;
		}

		for (int i = 0; i < player.field_7514.method_5439(); i++) {
			class_1799 invStack = player.field_7514.method_5438(i);

			if (invStack.method_7960() || !filter.test(invStack)) {
				continue;
			}

			if (Energy.valid(invStack)) {
				Energy.of(itemStack)
						.into(Energy.of(invStack))
						.move(maxOutput);
			}
		}
	}
}
