/*
 * This file is part of RebornCore, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2021 TeamReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.util;

import net.minecraft.class_1799;
import net.minecraft.class_2487;
import net.minecraft.class_2499;

public class ItemNBTHelper {

	/**
	 * Checks if an ItemStack has a Tag Compound
	 **/
	public static boolean detectNBT(class_1799 stack) {
		return stack.method_7985();
	}

	/**
	 * Tries to initialize an NBT Tag Compound in an ItemStack, this will not do
	 * anything if the stack already has a tag compound
	 **/
	public static void initNBT(class_1799 stack) {
		if (!detectNBT(stack)) {
			injectNBT(stack, new class_2487());
		}
	}

	/**
	 * Injects an NBT Tag Compound to an ItemStack, no checks are made
	 * previously
	 **/
	public static void injectNBT(class_1799 stack, class_2487 nbt) {
		stack.method_7980(nbt);
	}

	/**
	 * Gets the NBTTagCompound in an ItemStack. Tries to init it previously in
	 * case there isn't one present
	 **/
	public static class_2487 getNBT(class_1799 stack) {
		initNBT(stack);
		return stack.method_7969();
	}

	// SETTERS
	// ///////////////////////////////////////////////////////////////////

	public static void setBoolean(class_1799 stack, String tag, boolean b) {
		getNBT(stack).method_10556(tag, b);
	}

	public static void setByte(class_1799 stack, String tag, byte b) {
		getNBT(stack).method_10567(tag, b);
	}

	public static void setShort(class_1799 stack, String tag, short s) {
		getNBT(stack).method_10575(tag, s);
	}

	public static void setInt(class_1799 stack, String tag, int i) {
		getNBT(stack).method_10569(tag, i);
	}

	public static void setLong(class_1799 stack, String tag, long l) {
		getNBT(stack).method_10544(tag, l);
	}

	public static void setFloat(class_1799 stack, String tag, float f) {
		getNBT(stack).method_10548(tag, f);
	}

	public static void setDouble(class_1799 stack, String tag, double d) {
		getNBT(stack).method_10549(tag, d);
	}

	public static void setCompound(class_1799 stack, String tag, class_2487 cmp) {
		if (!tag.equalsIgnoreCase("ench")) // not override the enchantments
		{
			getNBT(stack).method_10566(tag, cmp);
		}
	}

	public static void setString(class_1799 stack, String tag, String s) {
		getNBT(stack).method_10582(tag, s);
	}

	public static void setList(class_1799 stack, String tag, class_2499 list) {
		getNBT(stack).method_10566(tag, list);
	}

	// GETTERS
	// ///////////////////////////////////////////////////////////////////

	public static boolean verifyExistance(class_1799 stack, String tag) {
		return !stack.method_7960() && getNBT(stack).method_10545(tag);
	}

	public static boolean getBoolean(class_1799 stack, String tag, boolean defaultExpected) {
		return verifyExistance(stack, tag) ? getNBT(stack).method_10577(tag) : defaultExpected;
	}

	public static byte getByte(class_1799 stack, String tag, byte defaultExpected) {
		return verifyExistance(stack, tag) ? getNBT(stack).method_10571(tag) : defaultExpected;
	}

	public static short getShort(class_1799 stack, String tag, short defaultExpected) {
		return verifyExistance(stack, tag) ? getNBT(stack).method_10568(tag) : defaultExpected;
	}

	public static int getInt(class_1799 stack, String tag, int defaultExpected) {
		return verifyExistance(stack, tag) ? getNBT(stack).method_10550(tag) : defaultExpected;
	}

	public static long getLong(class_1799 stack, String tag, long defaultExpected) {
		return verifyExistance(stack, tag) ? getNBT(stack).method_10537(tag) : defaultExpected;
	}

	public static float getFloat(class_1799 stack, String tag, float defaultExpected) {
		return verifyExistance(stack, tag) ? getNBT(stack).method_10583(tag) : defaultExpected;
	}

	public static double getDouble(class_1799 stack, String tag, double defaultExpected) {
		return verifyExistance(stack, tag) ? getNBT(stack).method_10574(tag) : defaultExpected;
	}

	/**
	 * If nullifyOnFail is true it'll return null if it doesn't find any
	 * compounds, otherwise it'll return a new one.
	 **/
	public static class_2487 getCompound(class_1799 stack, String tag, boolean nullifyOnFail) {
		return verifyExistance(stack, tag) ? getNBT(stack).method_10562(tag)
				: nullifyOnFail ? null : new class_2487();
	}

	public static String getString(class_1799 stack, String tag, String defaultExpected) {
		return verifyExistance(stack, tag) ? getNBT(stack).method_10558(tag) : defaultExpected;
	}

	public static class_2499 getList(class_1799 stack, String tag, int objtype, boolean nullifyOnFail) {
		return verifyExistance(stack, tag) ? getNBT(stack).method_10554(tag, objtype)
				: nullifyOnFail ? null : new class_2499();
	}
}
