/*
 * This file is part of RebornCore, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2021 TeamReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.recipes;

import reborncore.RebornCore;
import reborncore.api.recipe.IRecipeCrafterProvider;
import reborncore.common.blocks.BlockMachineBase;
import reborncore.common.crafting.RebornRecipe;
import reborncore.common.crafting.RebornRecipeType;
import reborncore.common.crafting.ingredient.RebornIngredient;
import reborncore.common.util.ItemUtils;
import reborncore.common.util.RebornInventory;
import team.reborn.energy.Energy;
import team.reborn.energy.EnergySide;
import team.reborn.energy.EnergyStorage;

import org.jetbrains.annotations.Nullable;
import java.util.ArrayList;
import java.util.Optional;
import net.minecraft.class_1799;
import net.minecraft.class_2338;
import net.minecraft.class_2487;
import net.minecraft.class_2586;
import net.minecraft.class_2680;

/**
 * Use this in your blockEntity entity to craft things
 */
public class RecipeCrafter implements IUpgradeHandler {

	/**
	 * This is the recipe type to use
	 */
	public RebornRecipeType<?> recipeType;

	/**
	 * This is the parent blockEntity
	 */
	public class_2586 blockEntity;

	/**
	 * This is the place to use the power from
	 */
	public EnergyStorage energy;

	public Optional<IUpgradeHandler> parentUpgradeHandler = Optional.empty();

	/**
	 * This is the amount of inputs that the setRecipe has
	 */
	public int inputs;

	/**
	 * This is the amount of outputs that the recipe has
	 */
	public int outputs;

	/**
	 * This is the inventory to use for the crafting
	 */
	public RebornInventory<?> inventory;

	/**
	 * This is the list of the slots that the crafting logic should look for the
	 * input item stacks.
	 */
	public int[] inputSlots;

	/**
	 * This is the list for the slots that the crafting logic should look fot
	 * the output item stacks.
	 */
	public int[] outputSlots;
	public RebornRecipe currentRecipe;
	public int currentTickTime = 0;
	public int currentNeededTicks = 1;// Set to 1 to stop rare crashes

	int ticksSinceLastChange;

	@Nullable
	public static ICrafterSoundHanlder soundHanlder = (firstRun, blockEntity) -> {
	};

	public RecipeCrafter(RebornRecipeType<?> recipeType, class_2586 blockEntity, int inputs, int outputs, RebornInventory<?> inventory,
						 int[] inputSlots, int[] outputSlots) {
		this.recipeType = recipeType;
		this.blockEntity = blockEntity;
		if (blockEntity instanceof EnergyStorage) {
			energy = (EnergyStorage) blockEntity;
		}
		if (blockEntity instanceof IUpgradeHandler) {
			parentUpgradeHandler = Optional.of((IUpgradeHandler) blockEntity);
		}
		this.inputs = inputs;
		this.outputs = outputs;
		this.inventory = inventory;
		this.inputSlots = inputSlots;
		this.outputSlots = outputSlots;
		if (!(blockEntity instanceof IRecipeCrafterProvider)) {
			RebornCore.LOGGER.error(blockEntity.getClass().getName() + " does not use IRecipeCrafterProvider report this to the issue tracker!");
		}
	}

	/**
	 * Call this on the blockEntity tick
	 */
	public void updateEntity() {
		if (blockEntity.method_10997() == null || blockEntity.method_10997().field_9236) {
			return;
		}
		ticksSinceLastChange++;
		// Force a has chanced every second
		if (ticksSinceLastChange == 20) {
			setInvDirty(true);
			ticksSinceLastChange = 0;
			setIsActive();
		}
		// It will now look for new recipes.
		if (currentRecipe == null && isInvDirty()) {
			updateCurrentRecipe();
		}
		if (currentRecipe != null) {
			// If it doesn't have all the inputs reset
			if (isInvDirty() && !hasAllInputs()) {
				currentRecipe = null;
				currentTickTime = 0;
				setIsActive();
			}
			// If it has reached the recipe tick time
			if (currentRecipe != null && currentTickTime >= currentNeededTicks && hasAllInputs()) {
				boolean canGiveInvAll = true;
				// Checks to see if it can fit the output
				for (int i = 0; i < currentRecipe.getOutputs().size(); i++) {
					if (!canFitOutput(currentRecipe.getOutputs().get(i), outputSlots[i])) {
						canGiveInvAll = false;
					}
				}
				// The slots that have been filled
				ArrayList<Integer> filledSlots = new ArrayList<>();
				if (canGiveInvAll && currentRecipe.onCraft(blockEntity)) {
					for (int i = 0; i < currentRecipe.getOutputs().size(); i++) {
						// Checks it has not been filled
						if (!filledSlots.contains(outputSlots[i])) {
							// Fills the slot with the output stack
							fitStack(currentRecipe.getOutputs().get(i).method_7972(), outputSlots[i]);
							filledSlots.add(outputSlots[i]);
						}
					}
					// This uses all the inputs
					useAllInputs();
					// Reset
					currentRecipe = null;
					currentTickTime = 0;
					updateCurrentRecipe();
					//Update active sate if the blockEntity isnt going to start crafting again
					if (currentRecipe == null) {
						setIsActive();
					}
				}
			} else if (currentRecipe != null && currentTickTime < currentNeededTicks) {
				double useRequirement = getEuPerTick(currentRecipe.getPower());
				if (Energy.of(energy).use(useRequirement)) {
					currentTickTime++;
					if ((currentTickTime == 1 || currentTickTime % 20 == 0) && soundHanlder != null) {
						soundHanlder.playSound(false, blockEntity);
					}
				}
			}
		}
		setInvDirty(false);
	}

	/**
	 * Checks that we have all inputs, can fit output and update max tick time and current tick time
	 */
	public void updateCurrentRecipe() {
		currentTickTime = 0;
		for (RebornRecipe recipe : recipeType.getRecipes(blockEntity.method_10997())) {
			// This checks to see if it has all of the inputs
			if (!hasAllInputs(recipe)) continue;
			if (!recipe.canCraft(blockEntity)) continue;

			// This checks to see if it can fit all of the outputs
			boolean hasOutputSpace = true;
			for (int i = 0; i < recipe.getOutputs().size(); i++) {
				if (!canFitOutput(recipe.getOutputs().get(i), outputSlots[i])) {
					hasOutputSpace = false;
				}
			}
			if (!hasOutputSpace) continue;
			// Sets the current recipe then syncs
			setCurrentRecipe(recipe);
			this.currentNeededTicks = Math.max((int) (currentRecipe.getTime() * (1.0 - getSpeedMultiplier())), 1);
			setIsActive();
			return;
		}
		setCurrentRecipe(null);
		currentNeededTicks = 0;
		setIsActive();
	}

	public boolean hasAllInputs() {
		return hasAllInputs(currentRecipe);
	}

	public boolean hasAllInputs(RebornRecipe recipeType) {
		if (recipeType == null) {
			return false;
		}
		for (RebornIngredient ingredient : recipeType.getRebornIngredients()) {
			boolean hasItem = false;
			for (int slot : inputSlots) {
				if (ingredient.test(inventory.method_5438(slot))) {
					hasItem = true;
				}
			}
			if (!hasItem) {
				return false;
			}
		}
		return true;
	}

	public void useAllInputs() {
		if (currentRecipe == null) {
			return;
		}
		for (RebornIngredient ingredient : currentRecipe.getRebornIngredients()) {
			for (int inputSlot : inputSlots) {// Uses all of the inputs
				if (ingredient.test(inventory.method_5438(inputSlot))) {
					inventory.shrinkSlot(inputSlot, ingredient.getCount());
					break;
				}
			}
		}
	}

	public boolean canFitOutput(class_1799 stack, int slot) {// Checks to see if it can fit the stack
		if (stack.method_7960()) {
			return true;
		}
		if (inventory.method_5438(slot).method_7960()) {
			return true;
		}
		if (ItemUtils.isItemEqual(inventory.method_5438(slot), stack, true, true)) {
			return stack.method_7947() + inventory.method_5438(slot).method_7947() <= stack.method_7914();
		}
		return false;
	}

	public void fitStack(class_1799 stack, int slot) {// This fits a stack into a slot
		if (stack.method_7960()) {
			return;
		}
		if (inventory.method_5438(slot).method_7960()) {// If the slot is empty set the contents
			inventory.method_5447(slot, stack);
			return;
		}
		if (ItemUtils.isItemEqual(inventory.method_5438(slot), stack, true)) {// If the slot has stuff in
			if (stack.method_7947() + inventory.method_5438(slot).method_7947() <= stack.method_7914()) {// Check to see if it fits
				class_1799 newStack = stack.method_7972();
				newStack.method_7939(inventory.method_5438(slot).method_7947() + stack.method_7947());// Sets
				// the
				// new
				// stack
				// size
				inventory.method_5447(slot, newStack);
			}
		}
	}

	public void read(class_2487 tag) {
		class_2487 data = tag.method_10562("Crater");

		if (data.method_10545("currentTickTime")) {
			currentTickTime = data.method_10550("currentTickTime");
		}

		if (blockEntity != null && blockEntity.method_10997() != null && blockEntity.method_10997().field_9236) {
			blockEntity.method_10997().method_8413(blockEntity.method_11016(),
					blockEntity.method_10997().method_8320(blockEntity.method_11016()),
					blockEntity.method_10997().method_8320(blockEntity.method_11016()), 3);
		}
	}

	public void write(class_2487 tag) {

		class_2487 data = new class_2487();

		data.method_10549("currentTickTime", currentTickTime);

		tag.method_10566("Crater", data);
	}

	private boolean isActive() {
		return currentRecipe != null && energy.getStored(EnergySide.UNKNOWN) >= currentRecipe.getPower();
	}

	public boolean canCraftAgain() {
		for (RebornRecipe recipe : recipeType.getRecipes(blockEntity.method_10997())) {
			if (recipe.canCraft(blockEntity) && hasAllInputs(recipe)) {
				for (int i = 0; i < recipe.getOutputs().size(); i++) {
					if (!canFitOutput(recipe.getOutputs().get(i), outputSlots[i])) {
						return false;
					}
				}
				return !(energy.getStored(EnergySide.UNKNOWN) < recipe.getPower());
			}
		}
		return false;
	}

	public void setIsActive() {
		class_2338 pos = blockEntity.method_11016();
		if (blockEntity.method_10997() == null) return;
		class_2680 oldState  = blockEntity.method_10997().method_8320(pos);
		if (oldState.method_26204() instanceof BlockMachineBase) {
			BlockMachineBase blockMachineBase = (BlockMachineBase) oldState.method_26204();
			boolean isActive = isActive() || canCraftAgain();

			if (isActive == oldState.method_11654(BlockMachineBase.ACTIVE)) {
				return;
			}

			blockMachineBase.setActive(isActive, blockEntity.method_10997(), pos);
			blockEntity.method_10997().method_8413(pos, oldState, blockEntity.method_10997().method_8320(pos), 3);
		}
	}

	public void setCurrentRecipe(RebornRecipe recipe) {
		this.currentRecipe = recipe;
	}

	public boolean isInvDirty() {
		return inventory.hasChanged();
	}

	public void setInvDirty(boolean isDiry) {
		inventory.setChanged(isDiry);
	}

	public boolean isStackValidInput(class_1799 stack) {
		if (stack.method_7960()) {
			return false;
		}

		//Test with a stack with the max stack size as some independents will check the stacksize. Bit of a hack but should work.
		class_1799 largeStack = stack.method_7972();
		largeStack.method_7939(largeStack.method_7914());
		for (RebornRecipe recipe : recipeType.getRecipes(blockEntity.method_10997())) {
			for (RebornIngredient ingredient : recipe.getRebornIngredients()) {
				if (ingredient.test(largeStack)) {
					return true;
				}
			}
		}
		return false;
	}

	@Override
	public void resetSpeedMulti() {
		parentUpgradeHandler.ifPresent(IUpgradeHandler::resetSpeedMulti);
	}

	@Override
	public double getSpeedMultiplier() {
		return Math.min(parentUpgradeHandler.map(IUpgradeHandler::getSpeedMultiplier).orElse(0D), 0.975);
	}

	@Override
	public void addPowerMulti(double amount) {
		parentUpgradeHandler.ifPresent(iUpgradeHandler -> iUpgradeHandler.addPowerMulti(amount));
	}

	@Override
	public void resetPowerMulti() {
		parentUpgradeHandler.ifPresent(IUpgradeHandler::resetPowerMulti);
	}

	@Override
	public double getPowerMultiplier() {
		return parentUpgradeHandler.map(IUpgradeHandler::getPowerMultiplier).orElse(1D);
	}

	@Override
	public double getEuPerTick(double baseEu) {
		double power = parentUpgradeHandler.map(iUpgradeHandler -> iUpgradeHandler.getEuPerTick(baseEu)).orElse(1D);
		return Math.min(power, energy.getMaxStoredPower());
	}

	@Override
	public void addSpeedMulti(double amount) {
		parentUpgradeHandler.ifPresent(iUpgradeHandler -> iUpgradeHandler.addSpeedMulti(amount));
	}
}
