/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.util;

import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.tuple.Pair;
import reborncore.client.screen.builder.Syncable;
import reborncore.common.blockentity.MachineBaseBlockEntity;
import reborncore.common.fluid.FluidValue;
import reborncore.common.fluid.container.FluidInstance;
import reborncore.common.fluid.container.GenericFluidContainer;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Supplier;
import net.minecraft.class_2350;
import net.minecraft.class_2378;
import net.minecraft.class_2487;
import net.minecraft.class_2960;
import net.minecraft.class_3611;
import net.minecraft.class_3612;

public class Tank implements GenericFluidContainer<class_2350>, Syncable {

	private final String name;
	@NotNull
	private FluidInstance fluidInstance = new FluidInstance();
	private final FluidValue capacity;

	@Nullable
	private class_2350 side = null;

	private final MachineBaseBlockEntity blockEntity;

	public Tank(String name, FluidValue capacity, MachineBaseBlockEntity blockEntity) {
		super();
		this.name = name;
		this.capacity = capacity;
		this.blockEntity = blockEntity;
	}

	@NotNull
	public FluidInstance getFluidInstance() {
		return getFluidInstance(side);
	}

	@NotNull
	public class_3611 getFluid() {
		return getFluidInstance().getFluid();
	}

	public FluidValue getCapacity() {
		return capacity;
	}

	public FluidValue getFreeSpace() {
		return getCapacity().subtract(getFluidAmount());
	}

	public boolean canFit(class_3611 fluid, FluidValue amount) {
		return (isEmpty() || getFluid() == fluid) && getFreeSpace().equalOrMoreThan(amount);
	}

	public boolean isEmpty() {
		return getFluidInstance().isEmpty();
	}

	public boolean isFull() {
		return !getFluidInstance().isEmpty() && getFluidInstance().getAmount().equalOrMoreThan(getCapacity());
	}

	public final class_2487 write(class_2487 nbt) {
		class_2487 tankData = fluidInstance.write();
		nbt.method_10566(name, tankData);
		return nbt;
	}

	public void setFluidAmount(FluidValue amount) {
		if (!fluidInstance.isEmptyFluid()) {
			fluidInstance.setAmount(amount);
		}
	}

	public final Tank read(class_2487 nbt) {
		if (nbt.method_10545(name)) {
			// allow to read empty tanks
			setFluid(class_3612.field_15906);

			class_2487 tankData = nbt.method_10562(name);
			fluidInstance = new FluidInstance(tankData);
		}
		return this;
	}

	public void setFluid(@NotNull class_3611 f) {
		Validate.notNull(f);
		fluidInstance.setFluid(f);
	}

	@Nullable
	public class_2350 getSide() {
		return side;
	}

	public void setSide(
			@Nullable
					class_2350 side) {
		this.side = side;
	}

	@Override
	public void getSyncPair(List<Pair<Supplier, Consumer>> pairList) {
		pairList.add(Pair.of(() -> class_2378.field_11154.method_10221(fluidInstance.getFluid()).toString(), (Consumer<String>) o -> fluidInstance.setFluid(class_2378.field_11154.method_10223(new class_2960(o)))));
		pairList.add(Pair.of(() -> fluidInstance.getAmount(), o -> fluidInstance.setAmount((FluidValue) o)));
	}

	public FluidValue getFluidAmount() {
		return getFluidInstance().getAmount();
	}

	@Override
	public void setFluid(@Nullable class_2350 type, @NotNull FluidInstance instance) {
		fluidInstance = instance;
	}

	@NotNull
	@Override
	public FluidInstance getFluidInstance(@Nullable class_2350 type) {
		return fluidInstance;
	}

	public void setFluidInstance(@NotNull FluidInstance fluidInstance) {
		this.fluidInstance = fluidInstance;
	}

	@Override
	public FluidValue getCapacity(@Nullable class_2350 type) {
		return capacity;
	}


}
