/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.powerSystem;

import org.jetbrains.annotations.Nullable;
import reborncore.api.IListInfoProvider;
import reborncore.common.blockentity.MachineBaseBlockEntity;
import reborncore.common.blockentity.RedstoneConfiguration;
import reborncore.common.util.StringUtils;
import team.reborn.energy.Energy;
import team.reborn.energy.EnergySide;
import team.reborn.energy.EnergyStorage;
import team.reborn.energy.EnergyTier;

import java.util.List;
import net.minecraft.class_124;
import net.minecraft.class_1799;
import net.minecraft.class_2350;
import net.minecraft.class_2487;
import net.minecraft.class_2561;
import net.minecraft.class_2586;
import net.minecraft.class_2588;
import net.minecraft.class_2591;
import net.minecraft.class_2680;
import net.minecraft.class_3532;

public abstract class PowerAcceptorBlockEntity extends MachineBaseBlockEntity implements EnergyStorage, IListInfoProvider // TechReborn
{
	private EnergyTier blockEntityPowerTier;
	private double energy;

	public double extraPowerStorage;
	public double extraPowerInput;
	public int extraTier;
	public double powerChange;
	public double powerLastTick;
	public boolean checkOverfill = true; // Set to false to disable the overfill check.

	public PowerAcceptorBlockEntity(class_2591<?> blockEntityType) {
		super(blockEntityType);
		checkTier();
	}

	public void checkTier() {
		if (this.getMaxInput(EnergySide.UNKNOWN) == 0) {
			blockEntityPowerTier = getTier((int) this.getBaseMaxOutput());
		} else {
			blockEntityPowerTier = getTier((int) this.getBaseMaxInput());
		}
	}

	// TO-DO: Move to Energy API probably. Cables use this method.
	public static EnergyTier getTier(int power) {
		for (EnergyTier tier : EnergyTier.values()) {
			if (tier.getMaxInput() >= power) {
				return tier;
			}
		}
		return EnergyTier.INFINITE;
	}

	/**
	 * Get amount of missing energy
	 *
	 * @return double Free space for energy in internal buffer
	 */
	public double getFreeSpace() {
		return getMaxStoredPower() - getStored(EnergySide.UNKNOWN);
	}

	/**
	 * Adds energy to block entity
	 *
	 * @param amount double Amount to add
	 */
	public void addEnergy(double amount){
		setStored(energy + amount);
	}

	/**
	 * Use energy from block entity
	 *
	 * @param amount double Amount of energy to use
	 */
	public void useEnergy(double amount){
		if (energy > amount) {
			setStored(energy - amount);
		} else {
			setStored(0);
		}
	}

	/**
	 * Charge machine from battery placed inside inventory slot
	 *
	 * @param slot int Slot ID for battery slot
	 */
	public void charge(int slot) {
		if (field_11863 == null) {
			return;
		}
		if (field_11863.field_9236) {
			return;
		}

		double chargeEnergy = Math.min(getFreeSpace(), getMaxInput(EnergySide.UNKNOWN));
		if (chargeEnergy <= 0.0) {
			return;
		}
		if (!getOptionalInventory().isPresent()) {
			return;
		}
		class_1799 batteryStack = getOptionalInventory().get().method_5438(slot);
		if (batteryStack.method_7960()) {
			return;
		}

		if (Energy.valid(batteryStack)) {
			Energy.of(batteryStack).into(Energy.of(this)).move();
		}
	}

	/**
	 * Charge battery placed inside inventory slot from machine
	 *
	 * @param slot int Slot ID for battery slot
	 */
	public void discharge(int slot) {
		if (field_11863 == null) {
			return;
		}

		if (field_11863.field_9236) {
			return;
		}

		if (!getOptionalInventory().isPresent()){
			return;
		}

		class_1799 batteryStack = getOptionalInventory().get().method_5438(slot);
		if (batteryStack.method_7960()) {
			return;
		}

		if (Energy.valid(batteryStack)) {
			Energy.of(this).into(Energy.of(batteryStack)).move(getTier().getMaxOutput());
		}
	}

	/**
	 * Calculates the comparator output of a powered BE with the formula
	 * {@code ceil(blockEntity.getStored(EnergySide.UNKNOWN) * 15.0 / storage.getMaxPower())}.
	 *
	 * @param blockEntity the powered BE
	 * @return the calculated comparator output or 0 if {@code blockEntity} is not a {@code PowerAcceptorBlockEntity}
	 */
	public static int calculateComparatorOutputFromEnergy(@Nullable class_2586 blockEntity) {
		if (blockEntity instanceof PowerAcceptorBlockEntity) {
			PowerAcceptorBlockEntity storage = (PowerAcceptorBlockEntity) blockEntity;
			return class_3532.method_15384(storage.getStored(EnergySide.UNKNOWN) * 15.0 / storage.getMaxStoredPower());
		} else {
			return 0;
		}
	}

	/**
	 * Check if machine should load energy data from NBT
	 *
	 * @return boolean Returns true if machine should load energy data from NBT
	 */
	protected boolean shouldHandleEnergyNBT() {
		return true;
	}

	/**
	 * Check if block entity can accept energy from a particular side
	 *
	 * @param side EnergySide Machine side
	 * @return boolean Returns true if machine can accept energy from side provided
	 */
	protected boolean canAcceptEnergy(EnergySide side){
		return true;
	}

	/**
	 * Check if block entity can provide energy via a particular side
	 *
	 * @param side EnergySide Machine side
	 * @return boolean Returns true if machine can provide energy via particular side
	 */
	protected boolean canProvideEnergy(EnergySide side){
		return true;
	}

	@Deprecated
	public boolean canAcceptEnergy(class_2350 direction) {
		return true;
	}

	@Deprecated
	public boolean canProvideEnergy(class_2350 direction) {
		return true;
	}

	/**
	 * Wrapper method used to sync additional energy storage values with client via BlockEntityScreenHandlerBuilder
	 *
	 * @return double Size of additional energy buffer
	 */
	public double getExtraPowerStorage(){
		return extraPowerStorage;
	}

	/**
	 * Wrapper method used to sync additional energy storage values with client via BlockEntityScreenHandlerBuilder
	 *
	 * @param extraPowerStorage double Size of additional energy buffer
	 */
	public void setExtraPowerStorage(double extraPowerStorage) {
		this.extraPowerStorage = extraPowerStorage;
	}

	/**
	 * Wrapper method used to sync energy change values with client via BlockEntityScreenHandlerBuilder
	 *
	 * @return double Energy change per tick
	 */
	public double getPowerChange() {
		return powerChange;
	}

	/**
	 * Wrapper method used to sync energy change values with client via BlockEntityScreenHandlerBuilder
	 *
	 * @param powerChange double Energy change per tick
	 */
	public void setPowerChange(double powerChange) {
		this.powerChange = powerChange;
	}

	/**
	 * Wrapper method used to sync energy values with client via BlockEntityScreenHandlerBuilder
	 *
	 * @return double Energy stored in block entity
	 */

	public double getEnergy() {
		return getStored(EnergySide.UNKNOWN);
	}

	/**
	 * Wrapper method used to sync energy values with client via BlockEntityScreenHandlerBuilder
	 * @param energy double Energy stored in block entity
	 */
	public void setEnergy(double energy) {
		setStored(energy);
	}

	/**
	 * Returns base size of internal Energy buffer of a particular machine before any upgrades applied
	 *
	 * @return double Size of internal Energy buffer
	 */
	public abstract double getBaseMaxPower();

	/**
	 * Returns base output rate or zero if machine doesn't output energy
	 *
	 * @return double Output rate, E\t
	 */
	public abstract double getBaseMaxOutput();

	/**
	 * Returns base input rate or zero if machine doesn't accept energy
	 *
	 * @return double Input rate, E\t
	 */
	public abstract double getBaseMaxInput();

	// MachineBaseBlockEntity
	@Override
	public void method_16896() {
		super.method_16896();
		if (field_11863 == null || field_11863.field_9236) {
			return;
		}
		if (getStored(EnergySide.UNKNOWN) <= 0) {
			return;
		}
		if (!isActive(RedstoneConfiguration.POWER_IO)) {
			return;
		}

		for (class_2350 side : class_2350.values()) {
			class_2586 blockEntity = field_11863.method_8321(method_11016().method_10093(side));
			if (blockEntity == null || !Energy.valid(blockEntity)) {
				continue;
			}
			Energy.of(this)
					.side(side)
					.into(Energy.of(blockEntity).side(side.method_10153()))
					.move();
		}

		powerChange = getStored(EnergySide.UNKNOWN) - powerLastTick;
		powerLastTick = getStored(EnergySide.UNKNOWN);
	}

	@Override
	public void method_11014(class_2680 blockState, class_2487 tag) {
		super.method_11014(blockState, tag);
		class_2487 data = tag.method_10562("PowerAcceptor");
		if (shouldHandleEnergyNBT()) {
			this.setStored(data.method_10574("energy"));
		}
	}

	@Override
	public class_2487 method_11007(class_2487 tag) {
		super.method_11007(tag);
		class_2487 data = new class_2487();
		data.method_10549("energy", getStored(EnergySide.UNKNOWN));
		tag.method_10566("PowerAcceptor", data);
		return tag;
	}

	@Override
	public void resetUpgrades() {
		super.resetUpgrades();
		extraPowerStorage = 0;
		extraTier = 0;
		extraPowerInput = 0;
	}

	// EnergyStorage
	@Override
	public double getStored(EnergySide face) {
		return energy;
	}

	@Override
	public void setStored(double amount) {
		this.energy = amount;
		if(checkOverfill){
			this.energy = Math.max(Math.min(energy, getMaxStoredPower()), 0);
		}
		method_5431();
	}

	@Override
	public double getMaxStoredPower() {
		return getBaseMaxPower() + extraPowerStorage;
	}

	@Override
	public double getMaxOutput(EnergySide face) {
		if (!isActive(RedstoneConfiguration.POWER_IO)) {
			return 0;
		}
		if(!canProvideEnergy(face)) {
			return 0;
		}
		if (this.extraTier > 0) {
			return this.getTier().getMaxOutput();
		}
		return getBaseMaxOutput();
	}

	@Override
	public double getMaxInput(EnergySide face) {
		if (!isActive(RedstoneConfiguration.POWER_IO)) {
			return 0;
		}
		if (!canAcceptEnergy(face)) {
			return 0;
		}
		if (this.extraTier > 0) {
			return this.getTier().getMaxInput();
		}
		return getBaseMaxInput() + extraPowerInput;
	}

	@Override
	public EnergyTier getTier() {
		if (blockEntityPowerTier == null) {
			checkTier();
		}

		if (extraTier > 0) {
			for (EnergyTier enumTier : EnergyTier.values()) {
				if (enumTier.ordinal() == blockEntityPowerTier.ordinal() + extraTier) {
					return enumTier;
				}
			}
			return EnergyTier.INFINITE;
		}
		return blockEntityPowerTier;
	}

	// IListInfoProvider
	@Override
	public void addInfo(List<class_2561> info, boolean isReal, boolean hasData) {
		if (!isReal && hasData) {
			info.add(
					new class_2588("reborncore.tooltip.energy")
							.method_27692(class_124.field_1080)
							.method_27693(": ")
							.method_27693(PowerSystem.getLocalizedPower(energy))
							.method_27692(class_124.field_1065)
			);
		}

		info.add(
				new class_2588("reborncore.tooltip.energy.maxEnergy")
				.method_27692(class_124.field_1080)
				.method_27693(": ")
				.method_27693(PowerSystem.getLocalizedPower(getMaxStoredPower()))
				.method_27692(class_124.field_1065)
		);

		if (getMaxInput(EnergySide.UNKNOWN) != 0) {
			info.add(
					new class_2588("reborncore.tooltip.energy.inputRate")
							.method_27692(class_124.field_1080)
							.method_27693(": ")
							.method_27693(PowerSystem.getLocalizedPower(getMaxInput(EnergySide.UNKNOWN)))
							.method_27692(class_124.field_1065)
			);
		}
		if (getMaxOutput(EnergySide.UNKNOWN) > 0) {
			info.add(
					new class_2588("reborncore.tooltip.energy.outputRate")
							.method_27692(class_124.field_1080)
							.method_27693(": ")
							.method_27693(PowerSystem.getLocalizedPower(getMaxOutput(EnergySide.UNKNOWN)))
							.method_27692(class_124.field_1065)
			);
		}

		info.add(
				new class_2588("reborncore.tooltip.energy.tier")
						.method_27692(class_124.field_1080)
						.method_27693(": ")
						.method_27693(StringUtils.toFirstCapitalAllLowercase(getTier().toString()))
						.method_27692(class_124.field_1065)
		);

		if (isReal) {
			info.add(
					new class_2588("reborncore.tooltip.energy.change")
							.method_27692(class_124.field_1080)
							.method_27693(": ")
							.method_27693(PowerSystem.getLocalizedPower(powerChange))
							.method_27693("/t")
							.method_27692(class_124.field_1065)
			);
		}



		super.addInfo(info, isReal, hasData);
	}
}
