/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.blockentity;

import reborncore.common.fluid.FluidUtil;
import reborncore.common.util.NBTSerializable;
import reborncore.common.util.Tank;

import org.jetbrains.annotations.NotNull;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2487;
import net.minecraft.class_2586;

public class FluidConfiguration implements NBTSerializable {

	HashMap<class_2350, FluidConfig> sideMap;
	boolean input, output;

	public FluidConfiguration() {
		sideMap = new HashMap<>();
		Arrays.stream(class_2350.values()).forEach(facing -> sideMap.put(facing, new FluidConfig(facing)));
	}

	public FluidConfiguration(class_2487 tagCompound) {
		sideMap = new HashMap<>();
		read(tagCompound);
	}

	public FluidConfig getSideDetail(class_2350 side) {
		if (side == null) {
			return sideMap.get(class_2350.field_11043);
		}
		return sideMap.get(side);
	}

	public List<FluidConfig> getAllSides() {
		return new ArrayList<>(sideMap.values());
	}

	public void updateFluidConfig(FluidConfig config) {
		FluidConfig toEdit = sideMap.get(config.side);
		toEdit.ioConfig = config.ioConfig;
	}

	public void update(MachineBaseBlockEntity machineBase) {
		if (!input && !output) {
			return;
		}
		if (machineBase.getTank() == null || machineBase.method_10997().method_8510() % machineBase.slotTransferSpeed() != 0) {
			return;
		}
		for (class_2350 facing : class_2350.values()) {
			FluidConfig fluidConfig = getSideDetail(facing);
			if (fluidConfig == null || !fluidConfig.getIoConfig().isEnabled()) {
				continue;
			}

			Tank tank = getTank(machineBase, facing);
			if (autoInput() && fluidConfig.getIoConfig().isInsert()) {
				FluidUtil.transferFluid(tank, machineBase.getTank(), machineBase.fluidTransferAmount());
			}
			if (autoOutput() && fluidConfig.getIoConfig().isExtact()) {
				FluidUtil.transferFluid(machineBase.getTank(), tank, machineBase.fluidTransferAmount());
			}
		}
	}

	private Tank getTank(MachineBaseBlockEntity machine, class_2350 facing) {
		class_2338 pos = machine.method_11016().method_10093(facing);
		class_2586 blockEntity = machine.method_10997().method_8321(pos);
		if (blockEntity instanceof MachineBaseBlockEntity) {
			return ((MachineBaseBlockEntity) blockEntity).getTank();
		}
		return null;
	}

	public boolean autoInput() {
		return input;
	}

	public boolean autoOutput() {
		return output;
	}

	public void setInput(boolean input) {
		this.input = input;
	}

	public void setOutput(boolean output) {
		this.output = output;
	}

	@NotNull
	@Override
	public class_2487 write() {
		class_2487 compound = new class_2487();
		Arrays.stream(class_2350.values()).forEach(facing -> compound.method_10566("side_" + facing.ordinal(), sideMap.get(facing).write()));
		compound.method_10556("input", input);
		compound.method_10556("output", output);
		return compound;
	}

	@Override
	public void read(@NotNull class_2487 nbt) {
		sideMap.clear();
		Arrays.stream(class_2350.values()).forEach(facing -> {
			class_2487 compound = nbt.method_10562("side_" + facing.ordinal());
			FluidConfig config = new FluidConfig(compound);
			sideMap.put(facing, config);
		});
		input = nbt.method_10577("input");
		output = nbt.method_10577("output");
	}

	public static class FluidConfig implements NBTSerializable {
		class_2350 side;
		FluidConfiguration.ExtractConfig ioConfig;

		public FluidConfig(class_2350 side) {
			this.side = side;
			this.ioConfig = ExtractConfig.ALL;
		}

		public FluidConfig(class_2350 side, FluidConfiguration.ExtractConfig ioConfig) {
			this.side = side;
			this.ioConfig = ioConfig;
		}

		public FluidConfig(class_2487 tagCompound) {
			read(tagCompound);
		}

		public class_2350 getSide() {
			return side;
		}

		public ExtractConfig getIoConfig() {
			return ioConfig;
		}

		@NotNull
		@Override
		public class_2487 write() {
			class_2487 tagCompound = new class_2487();
			tagCompound.method_10569("side", side.ordinal());
			tagCompound.method_10569("config", ioConfig.ordinal());
			return tagCompound;
		}

		@Override
		public void read(@NotNull class_2487 nbt) {
			side = class_2350.values()[nbt.method_10550("side")];
			ioConfig = FluidConfiguration.ExtractConfig.values()[nbt.method_10550("config")];
		}
	}

	public enum ExtractConfig {
		NONE(false, false),
		INPUT(false, true),
		OUTPUT(true, false),
		ALL(true, true);

		boolean extact;
		boolean insert;

		ExtractConfig(boolean extact, boolean insert) {
			this.extact = extact;
			this.insert = insert;
		}

		public boolean isExtact() {
			return extact;
		}

		public boolean isInsert() {
			return insert;
		}

		public boolean isEnabled() {
			return extact || insert;
		}

		public FluidConfiguration.ExtractConfig getNext() {
			int i = this.ordinal() + 1;
			if (i >= FluidConfiguration.ExtractConfig.values().length) {
				i = 0;
			}
			return FluidConfiguration.ExtractConfig.values()[i];
		}
	}
}
