/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.crafting.ingredient;

import com.google.gson.JsonObject;
import com.google.gson.JsonSyntaxException;
import org.apache.commons.lang3.Validate;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1856;
import net.minecraft.class_2378;
import net.minecraft.class_2960;
import net.minecraft.class_3494;
import net.minecraft.class_3518;
import net.minecraft.class_5323;

public class TagIngredient extends RebornIngredient {

	private final class_2960 tagIdentifier;
	private final class_3494<class_1792> tag;
	private final Optional<Integer> count;

	public TagIngredient(class_2960 tagIdentifier, class_3494<class_1792> tag, Optional<Integer> count) {
		super(IngredientManager.TAG_RECIPE_TYPE);
		this.tagIdentifier = tagIdentifier;
		this.tag = tag;
		this.count = count;
	}

	@Override
	public boolean test(class_1799 itemStack) {
		if (count.isPresent() && count.get() > itemStack.method_7947()) {
			return false;
		}
		return itemStack.method_7909().method_7855(tag);
	}

	@Override
	public class_1856 getPreview() {
		return class_1856.method_8106(tag);
	}

	@Override
	public List<class_1799> getPreviewStacks() {
		return tag.method_15138().stream().map(class_1799::new).peek(itemStack -> itemStack.method_7939(count.orElse(1))).collect(Collectors.toList());
	}

	public static RebornIngredient deserialize(JsonObject json) {
		Optional<Integer> count = Optional.empty();
		if (json.has("count")) {
			count = Optional.of(class_3518.method_15260(json, "count"));
		}

		if (json.has("server_sync")) {
			class_2960 tagIdent = new class_2960(class_3518.method_15265(json, "tag_identifier"));
			List<class_1792> items = new ArrayList<>();
			for (int i = 0; i < class_3518.method_15260(json, "items"); i++) {
				class_2960 identifier = new class_2960(class_3518.method_15265(json, "item_" + i));
				class_1792 item = class_2378.field_11142.method_10223(identifier);
				Validate.isTrue(item != class_1802.field_8162, "item cannot be air");
				items.add(item);
			}
			return new TagIngredient(tagIdent, new SimpleTag<>(items), count);
		}

		class_2960 identifier = new class_2960(class_3518.method_15265(json, "tag"));
		class_3494<class_1792> tag = class_5323.method_29223().method_29220().method_15193(identifier);
		if (tag == null) {
			throw new JsonSyntaxException("Unknown item tag '" + identifier + "'");
		}
		return new TagIngredient(identifier, tag, count);
	}

	@Override
	public JsonObject toJson() {
		//Tags are not synced across the server so we sync all the items
		JsonObject jsonObject = new JsonObject();
		jsonObject.addProperty("server_sync", true);

		class_1792[] items = tag.method_15138().toArray(new class_1792[0]);
		jsonObject.addProperty("items", items.length);
		for (int i = 0; i < items.length; i++) {
			jsonObject.addProperty("item_" + i, class_2378.field_11142.method_10221(items[i]).toString());
		}

		count.ifPresent(integer -> jsonObject.addProperty("count", integer));
		jsonObject.addProperty("tag_identifier", tagIdentifier.toString());
		return jsonObject;
	}

	@Override
	public int getCount() {
		return count.orElse(1);
	}
}
