/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.crafting;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.mojang.serialization.Dynamic;
import com.mojang.serialization.JsonOps;
import io.github.cottonmc.libcd.api.CustomOutputRecipe;
import org.apache.commons.lang3.Validate;
import reborncore.api.recipe.IRecipeCrafterProvider;
import reborncore.common.crafting.ingredient.DummyIngredient;
import reborncore.common.crafting.ingredient.IngredientManager;
import reborncore.common.crafting.ingredient.RebornIngredient;
import reborncore.common.util.DefaultedListCollector;
import reborncore.common.util.serialization.SerializationUtil;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import net.minecraft.class_1263;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1856;
import net.minecraft.class_1860;
import net.minecraft.class_1865;
import net.minecraft.class_1937;
import net.minecraft.class_2371;
import net.minecraft.class_2378;
import net.minecraft.class_2509;
import net.minecraft.class_2540;
import net.minecraft.class_2586;
import net.minecraft.class_2960;
import net.minecraft.class_3518;

public class RebornRecipe implements class_1860<class_1263>, CustomOutputRecipe {

	private final RebornRecipeType<?> type;
	private final class_2960 name;

	private class_2371<RebornIngredient> ingredients = class_2371.method_10211();
	private class_2371<class_1799> outputs = class_2371.method_10211();
	protected int power;
	protected int time;

	protected boolean dummy = false;

	public RebornRecipe(RebornRecipeType<?> type, class_2960 name) {
		this.type = type;
		this.name = name;
	}

	public RebornRecipe(RebornRecipeType<?> type, class_2960 name, class_2371<RebornIngredient> ingredients, class_2371<class_1799> outputs, int power, int time) {
		this(type, name);
		this.ingredients = ingredients;
		this.outputs = outputs;
		this.power = power;
		this.time = time;
	}

	public void deserialize(JsonObject jsonObject) {
		if (jsonObject.has("dummy")) {
			makeDummy();
			return;
		}

		//Crash if the recipe has all ready been deserialized
		Validate.isTrue(ingredients.isEmpty());

		power = class_3518.method_15260(jsonObject, "power");
		time = class_3518.method_15260(jsonObject, "time");

		ingredients = SerializationUtil.stream(class_3518.method_15261(jsonObject, "ingredients"))
				.map(IngredientManager::deserialize)
				.collect(DefaultedListCollector.toList());

		JsonArray resultsJson = class_3518.method_15261(jsonObject, "results");
		outputs = RecipeUtils.deserializeItems(resultsJson);
	}

	public void serialize(JsonObject jsonObject) {
		if (isDummy()) {
			jsonObject.addProperty("dummy", true);
			return;
		}
		jsonObject.addProperty("power", power);
		jsonObject.addProperty("time", time);

		JsonArray ingredientsArray = new JsonArray();
		getRebornIngredients().stream().map(RebornIngredient::witeToJson).forEach(ingredientsArray::add);
		jsonObject.add("ingredients", ingredientsArray);

		JsonArray resultsArray = new JsonArray();
		for (class_1799 stack : outputs) {
			JsonObject stackObject = new JsonObject();
			stackObject.addProperty("item", class_2378.field_11142.method_10221(stack.method_7909()).toString());
			if (stack.method_7947() > 1) {
				stackObject.addProperty("count", stack.method_7947());
			}
			if (stack.method_7985()) {
				stackObject.add("nbt", Dynamic.convert(class_2509.field_11560, JsonOps.INSTANCE, stack.method_7969()));
			}
			resultsArray.add(stackObject);
		}
		jsonObject.add("results", resultsArray);
	}

	public void serialize(class_2540 byteBuf) {

	}

	public void deserialize(class_2540 byteBuf) {

	}

	@Override
	public class_2960 method_8114() {
		return name;
	}

	@Override
	public class_1865<?> method_8119() {
		return type;
	}

	@Override
	public net.minecraft.class_3956<?> method_17716() {
		return type;
	}

	public RebornRecipeType<?> getRebornRecipeType() {
		return type;
	}

	// use the RebornIngredient version to ensure stack sizes are checked
	@Deprecated
	@Override
	public class_2371<class_1856> method_8117() {
		return ingredients.stream().map(RebornIngredient::getPreview).collect(DefaultedListCollector.toList());
	}

	public class_2371<RebornIngredient> getRebornIngredients() {
		return ingredients;
	}

	public List<class_1799> getOutputs() {
		return Collections.unmodifiableList(outputs);
	}

	public int getPower() {
		return power;
	}

	public int getTime() {
		return time;
	}

	/**
	 * @param blockEntity the blockEntity that is doing the crafting
	 * @return if true the recipe will craft, if false it will not
	 */
	public boolean canCraft(class_2586 blockEntity) {
		if (isDummy()) {
			return false;
		}
		if (blockEntity instanceof IRecipeCrafterProvider) {
			return ((IRecipeCrafterProvider) blockEntity).canCraft(this);
		}
		return true;
	}

	/**
	 * @param blockEntity the blockEntity that is doing the crafting
	 * @return return true if fluid was taken and should craft
	 */
	public boolean onCraft(class_2586 blockEntity) {
		return true;
	}

	//Done as our recipes do not support these functions, hopefully nothing blidly calls them

	@Deprecated
	@Override
	public boolean method_8115(class_1263 inv, class_1937 worldIn) {
		throw new UnsupportedOperationException();
	}

	@Deprecated
	@Override
	public class_1799 method_8116(class_1263 inv) {
		throw new UnsupportedOperationException();
	}

	@Deprecated
	@Override
	public boolean method_8113(int width, int height) {
		throw new UnsupportedOperationException();
	}

	@Deprecated
	@Override
	public class_1799 method_8110() {
		throw new UnsupportedOperationException();
	}

	@Override
	public class_2371<class_1799> method_8111(class_1263 p_179532_1_) {
		throw new UnsupportedOperationException();
	}

	//Done to try and stop the table from loading it
	@Override
	public boolean method_8118() {
		return true;
	}

	private boolean isDummy() {
		return dummy;
	}

	void makeDummy() {
		this.ingredients.add(new DummyIngredient());
		this.dummy = true;
	}

	@Override
	public Collection<class_1792> getOutputItems() {
		List<class_1792> items = new ArrayList<>();
		for (class_1799 stack : outputs) {
			items.add(stack.method_7909());
		}
		return items;
	}
}
