/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.crafting;

import java.util.HashMap;
import java.util.function.Function;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import org.apache.commons.lang3.Validate;

import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_2370;
import net.minecraft.class_2378;
import net.minecraft.class_2960;
import net.minecraft.class_3489;

public final class ConditionManager {

	private static final HashMap<class_2960, RecipeCondition<?>> RECIPE_CONDITIONS = new HashMap<>();
	private static final HashMap<class_2960, Class<?>> RECIPE_CONDITION_TYPES = new HashMap<>();

	static {
		//Only loads the recipe in a development env
		register("development", Boolean.class, (bool) -> bool == FabricLoader.getInstance().isDevelopmentEnvironment());

		//Only loads the recipe when the item is registered
		register("item", class_2960.class, (id) -> registryContains(class_2378.field_11142, id));

		//Only loads the recipe when the fluid is registered
		register("fluid", class_2960.class, (id) -> registryContains(class_2378.field_11154, id));

		//Only loads the recipe when the tag is loaded
		register("tag", class_2960.class, s -> class_3489.method_15106().method_15196().containsKey(s));

		//Only load the recipe if the provided mod is loaded
		register("mod", String.class, s -> FabricLoader.getInstance().isModLoaded(s));
	}

	private static boolean registryContains(class_2370<?> registry, class_2960 ident) {
		return registry.method_10250(ident);
	}

	public static <T> void register(String name, Class<T> type, RecipeCondition<T> recipeCondition){
		register(new class_2960("reborncore", name), type, recipeCondition);
	}

	public static <T> void register(class_2960 identifier, Class<T> type, RecipeCondition<T> recipeCondition){
		Validate.isTrue(!RECIPE_CONDITIONS.containsKey(identifier), "Recipe condition already registered");
		RECIPE_CONDITIONS.put(identifier, recipeCondition);
		RECIPE_CONDITION_TYPES.put(identifier, type);
	}

	public static RecipeCondition<?> getRecipeCondition(class_2960 identifier){
		RecipeCondition<?> condition = RECIPE_CONDITIONS.get(identifier);
		if(condition == null){
			throw new UnsupportedOperationException("Could not find recipe condition for " + identifier.toString());
		}
		return condition;
	}

	public static boolean shouldLoadRecipe(JsonObject jsonObject){
		if(!jsonObject.has("conditions")) return true;
		return jsonObject.get("conditions").getAsJsonObject().entrySet().stream()
			.allMatch(entry -> shouldLoad(entry.getKey(), entry.getValue()));
	}

	public static boolean shouldLoad(String ident, JsonElement jsonElement){
		class_2960 identifier = parseIdent(ident);
		RecipeCondition<?> recipeCondition = getRecipeCondition(identifier);
		Class<?> type = RECIPE_CONDITION_TYPES.get(identifier);
		return shouldLoad(type, jsonElement, recipeCondition);
	}

	@SuppressWarnings("unchecked")
	private static boolean shouldLoad(Class type, JsonElement jsonElement, RecipeCondition recipeCondition){
		Object val = TypeHelper.getValue(type, jsonElement);
		return recipeCondition.shouldLoad(val);
	}

	private static class_2960 parseIdent(String string) {
		if(string.contains(":")){
			return new class_2960(string);
		}
		return new class_2960("reborncore", string);
	}

	@FunctionalInterface
	public interface RecipeCondition<T> {
		boolean shouldLoad(T t);
	}

	private static class TypeHelper {

		private static final HashMap<Class<?>, Function<JsonElement, ?>> FUNCTIONS = new HashMap<>();

		static {
			register(String.class, JsonElement::getAsString);
			register(Boolean.class, JsonElement::getAsBoolean);

			register(class_2960.class, element -> new class_2960(element.getAsString()));
		}

		private static <T> void register(Class<T> type, Function<JsonElement, T> function){
			Validate.isTrue(!FUNCTIONS.containsKey(type), "Function for this class is already registered");
			FUNCTIONS.put(type, function);
		}

		public static <T> T getValue(Class<T> type, JsonElement jsonElement){
			Validate.isTrue(FUNCTIONS.containsKey(type), "Function for this class could not be found");
			//noinspection unchecked
			return (T) FUNCTIONS.get(type).apply(jsonElement);
		}

	}
}
