/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.blockentity;

import com.mojang.brigadier.exceptions.CommandSyntaxException;
import it.unimi.dsi.fastutil.ints.IntArrayList;
import it.unimi.dsi.fastutil.ints.IntList;
import it.unimi.dsi.fastutil.ints.IntLists;
import org.apache.commons.lang3.Validate;
import reborncore.RebornCore;
import reborncore.api.items.InventoryUtils;
import reborncore.common.util.ItemUtils;
import reborncore.common.util.NBTSerializable;
import reborncore.common.util.RebornInventory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import net.minecraft.class_1263;
import net.minecraft.class_1278;
import net.minecraft.class_1799;
import net.minecraft.class_2350;
import net.minecraft.class_2487;
import net.minecraft.class_2522;
import java.util.*;
import java.util.stream.Collectors;

public class SlotConfiguration implements NBTSerializable {

	List<SlotConfigHolder> slotDetails = new ArrayList<>();

	@Nullable
	class_1263 inventory;

	public SlotConfiguration(RebornInventory<?> inventory) {
		this.inventory = inventory;

		for (int i = 0; i < inventory.method_5439(); i++) {
			updateSlotDetails(new SlotConfigHolder(i));
		}
	}

	public void update(MachineBaseBlockEntity machineBase) {
		if (inventory == null && machineBase.getOptionalInventory().isPresent()) {
			inventory = machineBase.getOptionalInventory().get();
		}
		if (inventory != null && slotDetails.size() != inventory.method_5439()) {
			for (int i = 0; i < inventory.method_5439(); i++) {
				SlotConfigHolder holder = getSlotDetails(i);
				if (holder == null) {
					RebornCore.LOGGER.debug("Fixed slot " + i + " in " + machineBase);
					//humm somthing has gone wrong
					updateSlotDetails(new SlotConfigHolder(i));
				}
			}
		}
		if (!machineBase.method_10997().field_9236 && machineBase.method_10997().method_8510() % machineBase.slotTransferSpeed() == 0) {
			getSlotDetails().forEach(slotConfigHolder -> slotConfigHolder.handleItemIO(machineBase));
		}
	}

	public SlotConfiguration(class_2487 tagCompound) {
		read(tagCompound);
	}

	public List<SlotConfigHolder> getSlotDetails() {
		return slotDetails;
	}

	/**
	 * Replaces or adds a slot detail for the slot id
	 *
	 * @param slotConfigHolder
	 * @return SlotConfigHolder
	 */
	public SlotConfigHolder updateSlotDetails(SlotConfigHolder slotConfigHolder) {
		SlotConfigHolder lookup = getSlotDetails(slotConfigHolder.slotID);
		if (lookup != null) {
			slotDetails.remove(lookup);
		}
		slotDetails.add(slotConfigHolder);
		return slotConfigHolder;
	}

	@Nullable
	public SlotConfigHolder getSlotDetails(int id) {
		for (SlotConfigHolder detail : slotDetails) {
			if (detail.slotID == id) {
				return detail;
			}
		}
		return null;
	}

	public List<SlotConfig> getSlotsForSide(class_2350 facing) {
		return slotDetails.stream().map(slotConfigHolder -> slotConfigHolder.getSideDetail(facing)).collect(Collectors.toList());
	}

	@Nonnull
	@Override
	public class_2487 write() {
		class_2487 tagCompound = new class_2487();
		tagCompound.method_10569("size", slotDetails.size());
		for (int i = 0; i < slotDetails.size(); i++) {
			tagCompound.method_10566("slot_" + i, slotDetails.get(i).write());
		}
		return tagCompound;
	}

	@Override
	public void read(@Nonnull class_2487 nbt) {
		int size = nbt.method_10550("size");
		for (int i = 0; i < size; i++) {
			class_2487 tagCompound = nbt.method_10562("slot_" + i);
			SlotConfigHolder slotConfigHolder = new SlotConfigHolder(tagCompound);
			updateSlotDetails(slotConfigHolder);
		}
	}

	public static class SlotConfigHolder implements NBTSerializable {

		int slotID;
		HashMap<class_2350, SlotConfig> sideMap;
		boolean input, output, filter;

		public SlotConfigHolder(int slotID) {
			this.slotID = slotID;
			sideMap = new HashMap<>();
			Arrays.stream(class_2350.values()).forEach(facing -> sideMap.put(facing, new SlotConfig(facing, slotID)));
		}

		public SlotConfigHolder(class_2487 tagCompound) {
			sideMap = new HashMap<>();
			read(tagCompound);
			Validate.isTrue(Arrays.stream(class_2350.values())
				                .map(enumFacing -> sideMap.get(enumFacing))
				                .noneMatch(Objects::isNull),
			                "sideMap failed to load from nbt"
			);
		}

		public SlotConfig getSideDetail(class_2350 side) {
			Validate.notNull(side, "A none null side must be used");
			SlotConfig slotConfig = sideMap.get(side);
			Validate.notNull(slotConfig, "slotConfig was null for side " + side);
			return slotConfig;
		}

		public List<SlotConfig> getAllSides() {
			return new ArrayList<>(sideMap.values());
		}

		public void updateSlotConfig(SlotConfig config) {
			SlotConfig toEdit = sideMap.get(config.side);
			toEdit.slotIO = config.slotIO;
		}

		private void handleItemIO(MachineBaseBlockEntity machineBase) {
			if (!input && !output) {
				return;
			}
			getAllSides().stream()
				.filter(config -> config.getSlotIO().getIoConfig() != ExtractConfig.NONE)
				.forEach(config -> {
					if (input && config.getSlotIO().getIoConfig() == ExtractConfig.INPUT) {
						config.handleItemInput(machineBase);
					}
					if (output && config.getSlotIO().getIoConfig() == ExtractConfig.OUTPUT) {
						config.handleItemOutput(machineBase);
					}
				});
		}

		public boolean autoInput() {
			return input;
		}

		public boolean autoOutput() {
			return output;
		}

		public boolean filter() {
			return filter;
		}

		public void setInput(boolean input) {
			this.input = input;
		}

		public void setOutput(boolean output) {
			this.output = output;
		}

		public void setfilter(boolean filter) {
			this.filter = filter;
		}

		@Nonnull
		@Override
		public class_2487 write() {
			class_2487 compound = new class_2487();
			compound.method_10569("slotID", slotID);
			Arrays.stream(class_2350.values()).forEach(facing -> compound.method_10566("side_" + facing.ordinal(), sideMap.get(facing).write()));
			compound.method_10556("input", input);
			compound.method_10556("output", output);
			compound.method_10556("filter", filter);
			return compound;
		}

		@Override
		public void read(@Nonnull class_2487 nbt) {
			sideMap.clear();
			slotID = nbt.method_10550("slotID");
			Arrays.stream(class_2350.values()).forEach(facing -> {
				class_2487 compound = nbt.method_10562("side_" + facing.ordinal());
				SlotConfig config = new SlotConfig(compound);
				sideMap.put(facing, config);
			});
			input = nbt.method_10577("input");
			output = nbt.method_10577("output");
			if (nbt.method_10545("filter")) { //Was added later, this allows old saves to be upgraded
				filter = nbt.method_10577("filter");
			}
		}
	}

	public static class SlotConfig implements NBTSerializable {
		@Nonnull
		private class_2350 side;
		@Nonnull
		private SlotIO slotIO;
		private int slotID;

		public SlotConfig(@Nonnull class_2350 side, int slotID) {
			this.side = side;
			this.slotID = slotID;
			this.slotIO = new SlotIO(ExtractConfig.NONE);
		}

		public SlotConfig(@Nonnull class_2350 side, @Nonnull SlotIO slotIO, int slotID) {
			this.side = side;
			this.slotIO = slotIO;
			this.slotID = slotID;
		}

		public SlotConfig(class_2487 tagCompound) {
			read(tagCompound);
			Validate.notNull(side, "error when loading slot config");
			Validate.notNull(slotIO, "error when loading slot config");
		}

		@Nonnull
		public class_2350 getSide() {
			Validate.notNull(side);
			return side;
		}

		@Nonnull
		public SlotIO getSlotIO() {
			Validate.notNull(slotIO);
			return slotIO;
		}

		public int getSlotID() {
			return slotID;
		}

		private void handleItemInput(MachineBaseBlockEntity machineBase) {
			RebornInventory<?> inventory = machineBase.getOptionalInventory().get();
			class_1799 targetStack = inventory.method_5438(slotID);
			if (targetStack.method_7914() == targetStack.method_7947()) {
				return;
			}
			class_1263 sourceInv = InventoryUtils.getInventoryAt(machineBase.method_10997(), machineBase.method_11016().method_10093(side));
			if (sourceInv == null) {
				return;
			}

			IntList availableSlots = null;

			if (sourceInv instanceof class_1278) {
				availableSlots = IntArrayList.wrap(((class_1278) sourceInv).method_5494(side.method_10153()));
			}

			for (int i = 0; i < sourceInv.method_5439(); i++) {
				if (availableSlots != null && !availableSlots.contains(i)) {
					continue;
				}

				class_1799 sourceStack = sourceInv.method_5438(i);
				if (sourceStack.method_7960()) {
					continue;
				}
				if(!canInsertItem(slotID, sourceStack, side, machineBase)){
					continue;
				}

				if (sourceInv instanceof class_1278 && !((class_1278) sourceInv).method_5493(i, sourceStack, side.method_10153())) {
					continue;
				}

				//Checks if we are going to merge stacks that the items are the same
				if (!targetStack.method_7960()) {
					if (!ItemUtils.isItemEqual(sourceStack, targetStack, true, false)) {
						continue;
					}
				}
				int extract = 4;
				if (!targetStack.method_7960()) {
					extract = Math.min(targetStack.method_7914() - targetStack.method_7947(), extract);
				}
				class_1799 extractedStack = sourceInv.method_5434(i, extract);
				if (targetStack.method_7960()) {
					inventory.method_5447(slotID, extractedStack);
				} else {
					inventory.method_5438(slotID).method_7933(extractedStack.method_7947());
				}
				inventory.setChanged();
				break;
			}
		}

		private void handleItemOutput(MachineBaseBlockEntity machineBase) {
			RebornInventory<?> inventory = machineBase.getOptionalInventory().get();
			class_1799 sourceStack = inventory.method_5438(slotID);
			if (sourceStack.method_7960()) {
				return;
			}
			class_1263 destInventory = InventoryUtils.getInventoryAt(machineBase.method_10997(), machineBase.method_11016().method_10093(side));
			if (destInventory == null) {
				return;
			}

			class_1799 stack = InventoryUtils.insertItem(sourceStack, destInventory, side.method_10153());
			inventory.method_5447(slotID, stack);
		}

		@Nonnull
		@Override
		public class_2487 write() {
			class_2487 tagCompound = new class_2487();
			tagCompound.method_10569("side", side.ordinal());
			tagCompound.method_10566("config", slotIO.write());
			tagCompound.method_10569("slot", slotID);
			return tagCompound;
		}

		@Override
		public void read(@Nonnull class_2487 nbt) {
			side = class_2350.values()[nbt.method_10550("side")];
			slotIO = new SlotIO(nbt.method_10562("config"));
			slotID = nbt.method_10550("slot");
		}
	}

	public static class SlotIO implements NBTSerializable {
		ExtractConfig ioConfig;

		public SlotIO(class_2487 tagCompound) {
			read(tagCompound);
		}

		public SlotIO(ExtractConfig ioConfig) {
			this.ioConfig = ioConfig;
		}

		public ExtractConfig getIoConfig() {
			return ioConfig;
		}

		@Nonnull
		@Override
		public class_2487 write() {
			class_2487 compound = new class_2487();
			compound.method_10569("config", ioConfig.ordinal());
			return compound;
		}

		@Override
		public void read(@Nonnull class_2487 nbt) {
			ioConfig = ExtractConfig.values()[nbt.method_10550("config")];
		}
	}

	public enum ExtractConfig {
		NONE(false, false),
		INPUT(false, true),
		OUTPUT(true, false);

		boolean extact;
		boolean insert;

		ExtractConfig(boolean extact, boolean insert) {
			this.extact = extact;
			this.insert = insert;
		}

		public boolean isExtact() {
			return extact;
		}

		public boolean isInsert() {
			return insert;
		}

		public ExtractConfig getNext() {
			int i = this.ordinal() + 1;
			if (i >= ExtractConfig.values().length) {
				i = 0;
			}
			return ExtractConfig.values()[i];
		}
	}

	public String toJson(String machineIdent) {
		class_2487 tagCompound = new class_2487();
		tagCompound.method_10566("data", write());
		tagCompound.method_10582("machine", machineIdent);
		return tagCompound.toString();
	}

	public void readJson(String json, String machineIdent) throws UnsupportedOperationException {
		class_2487 compound;
		try {
			compound = class_2522.method_10718(json);
		} catch (CommandSyntaxException e) {
			throw new UnsupportedOperationException("Clipboard conetents isnt a valid slot configuation");
		}
		if (!compound.method_10545("machine") || !compound.method_10558("machine").equals(machineIdent)) {
			throw new UnsupportedOperationException("Machine config is not for this machine.");
		}
		read(compound.method_10562("data"));
	}

	//DO NOT CALL THIS, use the inventory access on the inventory
	public static boolean canInsertItem(int index, class_1799 itemStackIn, class_2350 direction, MachineBaseBlockEntity blockEntity) {
		if(itemStackIn.method_7960()){
			return false;
		}
		SlotConfiguration.SlotConfigHolder slotConfigHolder = blockEntity.getSlotConfiguration().getSlotDetails(index);
		SlotConfiguration.SlotConfig slotConfig = slotConfigHolder.getSideDetail(direction);
		if (slotConfig.getSlotIO().getIoConfig().isInsert()) {
			if (slotConfigHolder.filter()) {
				if(blockEntity instanceof SlotFilter){
					return ((SlotFilter) blockEntity).isStackValid(index, itemStackIn);
				}
			}
			return blockEntity.method_5437(index, itemStackIn);
		}
		return false;
	}

	//DO NOT CALL THIS, use the inventory access on the inventory
	public static boolean canExtractItem(int index, class_1799 stack, class_2350 direction, MachineBaseBlockEntity blockEntity) {
		SlotConfiguration.SlotConfigHolder slotConfigHolder = blockEntity.getSlotConfiguration().getSlotDetails(index);
		SlotConfiguration.SlotConfig slotConfig = slotConfigHolder.getSideDetail(direction);
		if (slotConfig.getSlotIO().getIoConfig().isExtact()) {
			return true;
		}
		return false;
	}

	public interface SlotFilter {
		boolean isStackValid(int slotID, class_1799 stack);

		int[] getInputSlots();
	}

}
