/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.crafting.ingredient;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonPrimitive;
import reborncore.common.fluid.container.ItemFluidInfo;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1856;
import net.minecraft.class_2378;
import net.minecraft.class_2960;
import net.minecraft.class_3518;
import net.minecraft.class_3528;
import net.minecraft.class_3611;
import net.minecraft.class_3612;

public class FluidIngredient extends RebornIngredient {

	private final class_3611 fluid;
	private final Optional<List<class_1792>> holders;
	private final Optional<Integer> count;

	private final class_3528<List<class_1799>> previewStacks;
	private final class_3528<class_1856> previewIngredient;

	public FluidIngredient(class_3611 fluid, Optional<List<class_1792>> holders, Optional<Integer> count) {
		super(IngredientManager.FLUID_RECIPE_TYPE);
		this.fluid = fluid;
		this.holders = holders;
		this.count = count;

		previewStacks = new class_3528<>(() -> class_2378.field_11142.method_10220()
			.filter(item -> item instanceof ItemFluidInfo)
			.filter(item -> !holders.isPresent() || holders.get().stream().anyMatch(i -> i == item))
			.map(item -> ((ItemFluidInfo)item).getFull(fluid))
			.peek(stack -> stack.method_7939(count.orElse(1)))
			.collect(Collectors.toList()));

		previewIngredient = new class_3528<>(() -> class_1856.method_8101(previewStacks.method_15332().toArray(new class_1799[0])));
	}

	public static RebornIngredient deserialize(JsonObject json) {
		class_2960 identifier = new class_2960(class_3518.method_15265(json, "fluid"));
		class_3611 fluid = class_2378.field_11154.method_10223(identifier);
		if(fluid == class_3612.field_15906){
			throw new JsonParseException("Fluid could not be found: " + class_3518.method_15265(json, "fluid"));
		}

		Optional<List<class_1792>> holders = Optional.empty();

		if(json.has("holder")){
			if(json.get("holder").isJsonPrimitive()){
				String ident = class_3518.method_15265(json, "holder");
				class_1792 item = class_2378.field_11142.method_10223(new class_2960(ident));
				if(item == class_1802.field_8162){
					throw new JsonParseException("could not find item:" + ident);
				}
				holders = Optional.of(Collections.singletonList(item));
			} else {
				JsonArray jsonArray = json.getAsJsonArray("holder");
				List<class_1792> itemList = new ArrayList<>();
				for (int i = 0; i < jsonArray.size(); i++) {
					String ident = jsonArray.get(i).getAsString();
					class_1792 item = class_2378.field_11142.method_10223(new class_2960(ident));
					if(item == class_1802.field_8162){
						throw new JsonParseException("could not find item:" + ident);
					}
					itemList.add(item);
				}
				holders = Optional.of(itemList);
			}
		}

		Optional<Integer> count = Optional.empty();

		if(json.has("count")){
			count = Optional.of(json.get("count").getAsInt());
		}

		return new FluidIngredient(fluid, holders, count);
	}

	@Override
	public boolean test(class_1799 itemStack) {
		if(holders.isPresent() && holders.get().stream().noneMatch(item -> itemStack.method_7909() == item)){
			return false;
		}
		if(count.isPresent() && itemStack.method_7947() < count.get()){
			return false;
		}
		if(itemStack.method_7909() instanceof ItemFluidInfo){
			return ((ItemFluidInfo) itemStack.method_7909()).getFluid(itemStack) == fluid;
		}
		return false;
	}

	@Override
	public class_1856 getPreview() {
		return previewIngredient.method_15332();
	}

	@Override
	public List<class_1799> getPreviewStacks() {
		return previewStacks.method_15332();
	}

	@Override
	public JsonObject toJson() {
		JsonObject jsonObject = new JsonObject();
		jsonObject.addProperty("fluid", class_2378.field_11154.method_10221(fluid).toString());
		if(holders.isPresent()){
			List<class_1792> holderList = holders.get();
			if(holderList.size() == 1){
				jsonObject.addProperty("holder", class_2378.field_11142.method_10221(holderList.get(0)).toString());
			} else {
				JsonArray holderArray = new JsonArray();
				holderList.forEach(item -> holderArray.add(new JsonPrimitive(class_2378.field_11142.method_10221(item).toString())));
				jsonObject.add("holder", holderArray);
			}
		}
		count.ifPresent(integer -> jsonObject.addProperty("count", integer));
		return jsonObject;
	}

	@Override
	public int getCount() {
		return count.orElse(1);
	}
}
