/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.crafting;

import com.google.gson.JsonObject;
import reborncore.common.crafting.ingredient.RebornIngredient;
import reborncore.common.fluid.FluidValue;
import reborncore.common.fluid.container.FluidInstance;
import reborncore.common.util.Tank;

import javax.annotation.Nonnull;
import net.minecraft.class_1799;
import net.minecraft.class_2371;
import net.minecraft.class_2378;
import net.minecraft.class_2586;
import net.minecraft.class_2960;
import net.minecraft.class_3518;
import net.minecraft.class_3611;

public abstract class RebornFluidRecipe extends RebornRecipe {

	@Nonnull
	private FluidInstance fluidInstance = FluidInstance.EMPTY;

	public RebornFluidRecipe(RebornRecipeType<?> type, class_2960 name) {
		super(type, name);
	}

	public RebornFluidRecipe(RebornRecipeType<?> type, class_2960 name, class_2371<RebornIngredient> ingredients, class_2371<class_1799> outputs, int power, int time) {
		super(type, name, ingredients, outputs, power, time);
	}

	public RebornFluidRecipe(RebornRecipeType<?> type, class_2960 name, class_2371<RebornIngredient> ingredients, class_2371<class_1799> outputs, int power, int time, FluidInstance fluidInstance) {
		this(type, name, ingredients, outputs, power, time);
		this.fluidInstance = fluidInstance;
	}

	@Override
	public void deserialize(JsonObject jsonObject) {
		super.deserialize(jsonObject);
		if(jsonObject.has("tank")){
			JsonObject tank = jsonObject.get("tank").getAsJsonObject();

			class_2960 identifier = new class_2960(class_3518.method_15265(tank, "fluid"));
			class_3611 fluid = class_2378.field_11154.method_10223(identifier);

			FluidValue value = FluidValue.BUCKET;
			if(tank.has("amount")){
				value = FluidValue.parseFluidValue(tank.get("amount"));
			}

			fluidInstance = new FluidInstance(fluid, value);
		}
	}

	public abstract Tank getTank(class_2586 be);

	@Override
	public boolean canCraft(class_2586 be) {
		final FluidInstance recipeFluid = fluidInstance;
		final FluidInstance tankFluid = getTank(be).getFluidInstance();
		if (fluidInstance.isEmpty()) {
			return true;
		}
		if (tankFluid.isEmpty()) {
			return false;
		}
		if (tankFluid.getFluid().equals(recipeFluid.getFluid())) {
			if (tankFluid.getAmount().equalOrMoreThan(recipeFluid.getAmount())) {
				return true;
			}
		}
		return false;
	}

	@Override
	public boolean onCraft(class_2586 be) {
		final FluidInstance recipeFluid = fluidInstance;
		final FluidInstance tankFluid = getTank(be).getFluidInstance();
		if (fluidInstance.isEmpty()) {
			return true;
		}
		if (tankFluid.isEmpty()) {
			return false;
		}
		if (tankFluid.getFluid().equals(recipeFluid.getFluid())) {
			if (tankFluid.getAmount().equalOrMoreThan(recipeFluid.getAmount())) {
				tankFluid.subtractAmount(recipeFluid.getAmount());
				return true;
			}
		}
		return false;
	}

	@Nonnull
	public FluidInstance getFluidInstance() {
		return fluidInstance;
	}
}
