/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.chunkloading;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import reborncore.common.network.ClientBoundPackets;
import reborncore.common.util.NBTSerializable;
import reborncore.common.world.DataAttachment;
import reborncore.common.world.DataAttachmentProvider;

import javax.annotation.Nonnull;
import net.minecraft.class_1923;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2378;
import net.minecraft.class_2487;
import net.minecraft.class_2499;
import net.minecraft.class_2960;
import net.minecraft.class_3215;
import net.minecraft.class_3218;
import net.minecraft.class_3222;
import net.minecraft.class_3230;
import java.util.*;
import java.util.stream.Collectors;

//This does not do the actual chunk loading, just keeps track of what chunks the chunk loader has loaded
public class ChunkLoaderManager implements DataAttachment {

	private static final class_3230<class_1923> CHUNK_LOADER = class_3230.method_14291("reborncore:chunk_loader", Comparator.comparingLong(class_1923::method_8324));

	public static ChunkLoaderManager get(class_1937 world){
		return DataAttachmentProvider.get(world, ChunkLoaderManager.class);
	}

	private final List<LoadedChunk> loadedChunks = new ArrayList<>();

	@Override
	public @Nonnull
	class_2487 write() {
		class_2487 tag = new class_2487();
		class_2499 listTag = new class_2499();

		listTag.addAll(loadedChunks.stream().map(LoadedChunk::write).collect(Collectors.toList()));
		tag.method_10566("loadedchunks", listTag);

		return tag;
	}

	@Override
	public void read(@Nonnull class_2487 tag) {
		loadedChunks.clear();
		class_2499 listTag = tag.method_10554("loadedchunks", tag.method_10711());

		loadedChunks.addAll(listTag.stream()
			                    .map(tag1 -> (class_2487) tag1)
			                    .map(LoadedChunk::new)
			                    .collect(Collectors.toList())
		);
	}

	public Optional<LoadedChunk> getLoadedChunk(class_1937 world, class_1923 chunkPos, class_2338 chunkLoader){
		return loadedChunks.stream()
			.filter(loadedChunk -> loadedChunk.getWorld().equals(getWorldName(world)))
			.filter(loadedChunk -> loadedChunk.getChunk().equals(chunkPos))
			.filter(loadedChunk -> loadedChunk.getChunkLoader().equals(chunkLoader))
			.findFirst();
	}

	public Optional<LoadedChunk> getLoadedChunk(class_1937 world, class_1923 chunkPos){
		return loadedChunks.stream()
			.filter(loadedChunk -> loadedChunk.getWorld().equals(getWorldName(world)))
			.filter(loadedChunk -> loadedChunk.getChunk().equals(chunkPos))
			.findFirst();
	}

	public List<LoadedChunk> getLoadedChunks(class_1937 world, class_2338 chunkloader){
		return loadedChunks.stream()
			.filter(loadedChunk -> loadedChunk.getWorld().equals(getWorldName(world)))
			.filter(loadedChunk -> loadedChunk.getChunkLoader().equals(chunkloader))
			.collect(Collectors.toList());
	}

	public boolean isChunkLoaded(class_1937 world, class_1923 chunkPos, class_2338 chunkLoader){
		return getLoadedChunk(world, chunkPos, chunkLoader).isPresent();
	}

	public boolean isChunkLoaded(class_1937 world, class_1923 chunkPos){
		return getLoadedChunk(world, chunkPos).isPresent();
	}


	public void loadChunk(class_1937 world, class_1923 chunkPos, class_2338 chunkLoader, String player){
		Validate.isTrue(!isChunkLoaded(world, chunkPos, chunkLoader), "chunk is already loaded");
		LoadedChunk loadedChunk = new LoadedChunk(chunkPos, getWorldName(world), player, chunkLoader);
		loadedChunks.add(loadedChunk);

		final class_3215 serverChunkManager = ((class_3218) world).method_14178();
		serverChunkManager.method_17297(ChunkLoaderManager.CHUNK_LOADER, loadedChunk.getChunk(), 31, loadedChunk.getChunk());

	}

	public void unloadChunkLoader(class_1937 world, class_2338 chunkLoader){
		getLoadedChunks(world, chunkLoader).forEach(loadedChunk -> unloadChunk(world, loadedChunk.getChunk(), chunkLoader));
	}

	public void unloadChunk(class_1937 world, class_1923 chunkPos, class_2338 chunkLoader){
		Optional<LoadedChunk> optionalLoadedChunk = getLoadedChunk(world, chunkPos, chunkLoader);
		Validate.isTrue(optionalLoadedChunk.isPresent(), "chunk is not loaded");

		LoadedChunk loadedChunk = optionalLoadedChunk.get();

		loadedChunks.remove(loadedChunk);

		if(!isChunkLoaded(world, loadedChunk.getChunk())){
			final class_3215 serverChunkManager = ((class_3218) world).method_14178();
			serverChunkManager.method_17300(ChunkLoaderManager.CHUNK_LOADER, loadedChunk.getChunk(), 31, loadedChunk.getChunk());
		}
	}

	public static class_2960 getWorldName(class_1937 world){
		Validate.isTrue(world instanceof class_3218, "world must be a ServerWorld");
		return class_2378.field_11155.method_10221(world.method_8597().method_12460());
	}

	public void syncChunkLoaderToClient(class_3222 serverPlayerEntity, class_2338 chunkLoader){
		syncToClient(serverPlayerEntity, loadedChunks.stream().filter(loadedChunk -> loadedChunk.getChunkLoader().equals(chunkLoader)).collect(Collectors.toList()));
	}

	public void syncAllToClient(class_3222 serverPlayerEntity) {
		syncToClient(serverPlayerEntity, loadedChunks);
	}

	public void clearClient(class_3222 serverPlayerEntity) {
		syncToClient(serverPlayerEntity, Collections.emptyList());
	}

	public void syncToClient(class_3222 serverPlayerEntity, List<LoadedChunk> chunks){
		serverPlayerEntity.field_13987.method_14364(
			ClientBoundPackets.createPacketSyncLoadedChunks(chunks)
		);
	}

	public static class LoadedChunk implements NBTSerializable {
		private class_1923 chunk;
		private class_2960 world;
		private String player;
		private class_2338 chunkLoader;

		public LoadedChunk(class_1923 chunk, class_2960 world, String player, class_2338 chunkLoader) {
			this.chunk = chunk;
			this.world = world;
			this.player = player;
			this.chunkLoader = chunkLoader;
			Validate.isTrue(!StringUtils.isBlank(player), "Player cannot be blank");
		}

		public LoadedChunk(class_2487 tag) {
			read(tag);
		}

		@Override
		public @Nonnull class_2487 write() {
			class_2487 tag = new class_2487();
			tag.method_10544("chunk", chunk.method_8324());
			tag.method_10582("world", world.toString());
			tag.method_10582("player", player);
			tag.method_10544("chunkLoader", chunkLoader.method_10063());
			return tag;
		}

		@Override
		public void read(@Nonnull class_2487 tag) {
			chunk = new class_1923(tag.method_10537("chunk"));
			world = new class_2960(tag.method_10558("world"));
			player = tag.method_10558("player");
			chunkLoader = class_2338.method_10092(tag.method_10537("chunkLoader"));
		}

		public class_1923 getChunk() {
			return chunk;
		}

		public class_2960 getWorld() {
			return world;
		}

		public String getPlayer() {
			return player;
		}

		public class_2338 getChunkLoader() {
			return chunkLoader;
		}
	}
}
