/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.powerSystem;

import reborncore.api.IListInfoProvider;
import reborncore.common.blockentity.MachineBaseBlockEntity;
import reborncore.common.blockentity.RedstoneConfiguration;
import reborncore.common.util.StringUtils;
import team.reborn.energy.Energy;
import team.reborn.energy.EnergySide;
import team.reborn.energy.EnergyStorage;
import team.reborn.energy.EnergyTier;

import javax.annotation.Nullable;
import net.minecraft.class_124;
import net.minecraft.class_1799;
import net.minecraft.class_2350;
import net.minecraft.class_2487;
import net.minecraft.class_2561;
import net.minecraft.class_2586;
import net.minecraft.class_2588;
import net.minecraft.class_2591;
import net.minecraft.class_2680;
import net.minecraft.class_3532;
import java.util.List;

public abstract class PowerAcceptorBlockEntity extends MachineBaseBlockEntity implements EnergyStorage, IListInfoProvider // TechReborn
{
	private EnergyTier blockEntityPowerTier;
	private double energy;

	public double extraPowerStorage;
	public double extraPowerInput;
	public int extraTier;
	public double powerChange;
	public double powerLastTick;
	public boolean checkOverfill = true; // Set to false to disable the overfill check.
	// Some external power systems (EU) support multiple energy packets per tick,
	// this allows machines to possibly emit
	// multiple packets in a tick. Other power systems such as FE will ignore this
	// option.
	public int maxPacketsPerTick = 1;

	public PowerAcceptorBlockEntity(class_2591<?> blockEntityType) {
		super(blockEntityType);
		checkTier();
	}

	public void checkTier() {
		if (this.getMaxInput(EnergySide.UNKNOWN) == 0) {
			blockEntityPowerTier = getTier((int) this.getBaseMaxOutput());
		} else {
			blockEntityPowerTier = getTier((int) this.getBaseMaxInput());
		}

	}

	public static EnergyTier getTier(int power) {
		for (EnergyTier tier : EnergyTier.values()) {
			if (tier.getMaxInput() >= power) {
				return tier;
			}
		}
		return EnergyTier.INFINITE;
	}

	public void setExtraPowerStorage(double extraPowerStorage) {
		this.extraPowerStorage = extraPowerStorage;
	}

	public void setMaxPacketsPerTick(int maxPacketsPerTick) {
		this.maxPacketsPerTick = maxPacketsPerTick;
	}

	public double getFreeSpace() {
		return getMaxPower() - getEnergy();
	}

	/**
	 * Charge machine from battery placed inside inventory slot
	 *
	 * @param slot int Slot ID for battery slot
	 */
	public void charge(int slot) {
		if (field_11863.field_9236) {
			return;
		}

		double chargeEnergy = Math.min(getFreeSpace(), getMaxInput(EnergySide.UNKNOWN));
		if (chargeEnergy <= 0.0) {
			return;
		}
		if (!getOptionalInventory().isPresent()) {
			return;
		}
		class_1799 batteryStack = getOptionalInventory().get().method_5438(slot);
		if (batteryStack.method_7960()) {
			return;
		}

		if (Energy.valid(batteryStack)) {
			Energy.of(batteryStack)
				.into(
					Energy
						.of(this)
				)
				.move();
		}

	}

	/**
	 * Charge battery from machine placed inside inventory slot
	 *
	 * @param slot int Slot ID for battery slot
	 */
	public void discharge(int slot) {
		if (field_11863.field_9236) {
			return;
		}

		class_1799 batteryStack = getOptionalInventory().get().method_5438(slot);
		if(batteryStack.method_7960()){
			return;
		}

		if(Energy.valid(batteryStack)){
			Energy.of(this).into(Energy.of(batteryStack)).move();
		}
	}

	public int getEnergyScaled(int scale) {
		return (int) ((getEnergy() * scale / getMaxPower()));
	}

	public boolean shouldHanldeEnergyNBT() {
		return true;
	}

	public boolean handleTierWithPower() {
		return true;
	}

	public double getPowerChange() {
		return powerChange;
	}

	public void setPowerChange(double powerChange) {
		this.powerChange = powerChange;
	}

	// TileMachineBase
	@Override
	public void method_16896() {
		super.method_16896();
		if (field_11863.field_9236) {
			return;
		}

		if (getEnergy() > 0 && isActive(RedstoneConfiguration.POWER_IO)) { // Tesla or IC2 should handle this if enabled, so only do this without tesla
			for (class_2350 side : class_2350.values()) {
				class_2586 blockEntity = method_10997().method_8321(method_11016().method_10093(side));
				if(blockEntity == null || !Energy.valid(blockEntity)){
					continue;
				}
				Energy.of(this)
					.side(side)
					.into(
						Energy.of(blockEntity).side(side.method_10153())
					)
					.move();
			}
		}

		powerChange = getEnergy() - powerLastTick;
		powerLastTick = getEnergy();
	}

	@Override
	public void method_11014(class_2680 blockState, class_2487 tag) {
		super.method_11014(blockState, tag);
		class_2487 data = tag.method_10562("PowerAcceptor");
		if (shouldHanldeEnergyNBT()) {
			this.setEnergy(data.method_10574("energy"));
		}
	}

	@Override
	public class_2487 method_11007(class_2487 tag) {
		super.method_11007(tag);
		class_2487 data = new class_2487();
		data.method_10549("energy", getEnergy());
		tag.method_10566("PowerAcceptor", data);
		return tag;
	}

	@Override
	public void resetUpgrades() {
		super.resetUpgrades();
		extraPowerStorage = 0;
		extraTier = 0;
		extraPowerInput = 0;
	}

//	@Override
//	public <T> LazyOptional<T> getCapability(Capability<T> capability, Direction facing) {
//		LazyOptional<T> externalCap = powerManagers.stream()
//			.filter(Objects::nonNull)
//			.map(externalPowerHandler -> externalPowerHandler.getCapability(capability, facing))
//			.filter(LazyOptional::isPresent)
//			.findFirst()
//			.orElse(null);
//
//		if (externalCap != null) {
//			return externalCap;
//		}
//
//		return super.getCapability(capability, facing);
//	}

	public abstract double getBaseMaxPower();

	public abstract double getBaseMaxOutput();

	public abstract double getBaseMaxInput();

	public double getEnergy() {
		return getStored(EnergySide.UNKNOWN);
	}

	@Override
	public double getStored(EnergySide face) {
		return energy;
	}

	@Override
	public void setStored(double amount) {
		this.energy = amount;
		if(checkOverfill){
			this.energy = Math.max(Math.min(energy, getMaxPower()), 0);
		}
		method_5431();
	}

	public void setEnergy(double energy) {
		setStored(energy);
	}

	public double addEnergy(double energy) {
		return addEnergy(energy, false);
	}

	public double addEnergy(double energy, boolean simulate) {
		double energyReceived = Math.min(getMaxPower(), Math.min(getFreeSpace(), energy));

		if (!simulate) {
			setEnergy(getEnergy() + energyReceived);
		}
		return energyReceived;
	}

	public boolean canUseEnergy(double input) {
		return input <= energy;
	}

	public double useEnergy(double energy) {
		return useEnergy(energy, false);
	}

	public double useEnergy(double extract, boolean simulate) {
		if (extract > energy) {
			extract = energy;
		}
		if (!simulate) {
			setEnergy(energy - extract);
		}
		return extract;
	}

	public boolean canAddEnergy(double energyIn) {
		return getEnergy() + energyIn <= getMaxPower();
	}

	public double getMaxPower() {
		return getBaseMaxPower() + extraPowerStorage;
	}

	@Override
	public double getMaxStoredPower() {
		return getMaxPower();
	}

	public boolean canAcceptEnergy(class_2350 direction) {
		return true;
	}

	public boolean canProvideEnergy(class_2350 direction) {
		return true;
	}

	@Override
	public double getMaxOutput(EnergySide face) {
		if (!isActive(RedstoneConfiguration.POWER_IO)) {
			return 0;
		}
		if(!canProvideEnergy(fromSide(face))) {
			return 0;
		}
		double maxOutput = 0;
		if (this.extraTier > 0) {
			maxOutput = this.getTier().getMaxOutput();
		} else {
			maxOutput = getBaseMaxOutput();
		}
		return maxOutput;
	}

	@Override
	public double getMaxInput(EnergySide face) {
		if (!isActive(RedstoneConfiguration.POWER_IO)) {
			return 0;
		}
		if(!canAcceptEnergy(fromSide(face))) {
			return 0;
		}
		double maxInput = 0;
		if (this.extraTier > 0) {
			maxInput = this.getTier().getMaxInput();
		} else {
			maxInput = getBaseMaxInput();
		}
		return maxInput + extraPowerInput;
	}

	public static class_2350 fromSide(EnergySide side){
		if(side == EnergySide.UNKNOWN){
			return null;
		}
		return class_2350.values()[side.ordinal()];
	}

	public EnergyTier getPushingTier() {
		return getTier();
	}

	@Override
	public EnergyTier getTier() {
		if (blockEntityPowerTier == null) {
			checkTier();
		}

		if (extraTier > 0) {
			for (EnergyTier enumTier : EnergyTier.values()) {
				if (enumTier.ordinal() == blockEntityPowerTier.ordinal() + extraTier) {
					return blockEntityPowerTier;
				}
			}
			return EnergyTier.INFINITE;
		}
		return blockEntityPowerTier;
	}

	// IListInfoProvider
	@Override
	public void addInfo(List<class_2561> info, boolean isReal, boolean hasData) {
		info.add(
				new class_2588("reborncore.tooltip.energy.maxEnergy")
				.method_27692(class_124.field_1080)
				.method_27693(": ")
				.method_27693(PowerSystem.getLocaliszedPowerFormatted(getMaxPower()))
				.method_27692(class_124.field_1065)
		);

		if (getMaxInput(EnergySide.UNKNOWN) != 0) {
			info.add(
					new class_2588("reborncore.tooltip.energy.inputRate")
							.method_27692(class_124.field_1080)
							.method_27693(": ")
							.method_27693(PowerSystem.getLocaliszedPowerFormatted(getMaxInput(EnergySide.UNKNOWN)))
							.method_27692(class_124.field_1065)
			);
		}
		if (getMaxOutput(EnergySide.UNKNOWN) <= 0) {
			info.add(
					new class_2588("reborncore.tooltip.energy.outputRate")
							.method_27692(class_124.field_1080)
							.method_27693(": ")
							.method_27693(PowerSystem.getLocaliszedPowerFormatted(getMaxOutput(EnergySide.UNKNOWN)))
							.method_27692(class_124.field_1065)
			);
		}

		info.add(
				new class_2588("reborncore.tooltip.energy.tier")
						.method_27692(class_124.field_1080)
						.method_27693(": ")
						.method_27693(StringUtils.toFirstCapitalAllLowercase(getTier().toString()))
						.method_27692(class_124.field_1065)
		);

		if (isReal) {
			new class_2588("reborncore.tooltip.energy.change")
					.method_27692(class_124.field_1080)
					.method_27693(": ")
					.method_27693(PowerSystem.getLocaliszedPowerFormatted(getPowerChange()))
					.method_27693("/t")
					.method_27692(class_124.field_1065);
		}

		if (hasData) {
			new class_2588("reborncore.tooltip.energy.change")
					.method_27692(class_124.field_1080)
					.method_27693(": ")
					.method_27693(PowerSystem.getLocaliszedPowerFormatted(energy))
					.method_27692(class_124.field_1065);
		}

		super.addInfo(info, isReal, hasData);
	}

	/**
	 * Calculates the comparator output of a powered BE with the formula
	 * {@code ceil(blockEntity.getEnergy() * 15.0 / storage.getMaxPower())}.
	 *
	 * @param blockEntity the powered BE
	 * @return the calculated comparator output or 0 if {@code blockEntity} is not a {@code PowerAcceptorBlockEntity}
	 */
	public static int calculateComparatorOutputFromEnergy(@Nullable class_2586 blockEntity) {
		if (blockEntity instanceof PowerAcceptorBlockEntity) {
			PowerAcceptorBlockEntity storage = (PowerAcceptorBlockEntity) blockEntity;
			return class_3532.method_15384(storage.getEnergy() * 15.0 / storage.getMaxPower());
		} else {
			return 0;
		}
	}

}
