/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.common.crafting;

import io.netty.buffer.Unpooled;
import org.apache.commons.lang3.Validate;
import reborncore.common.crafting.ingredient.RebornIngredient;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.stream.Collectors;
import net.minecraft.class_1799;
import net.minecraft.class_1860;
import net.minecraft.class_1865;
import net.minecraft.class_1937;
import net.minecraft.class_2378;
import net.minecraft.class_2540;
import net.minecraft.class_2960;

public class RecipeManager {

	private static final Map<class_2960, RebornRecipeType<?>> recipeTypes = new HashMap<>();

	public static <R extends RebornRecipe> RebornRecipeType<R> newRecipeType(BiFunction<RebornRecipeType<R>, class_2960, R> recipeFunction, class_2960 name){
		if(recipeTypes.containsKey(name)){
			throw new RuntimeException("RebornRecipe type with this name already registered");
		}
		RebornRecipeType<R> type = new RebornRecipeType<>(recipeFunction, name);
		recipeTypes.put(name, type);

		class_2378.method_10230(class_2378.field_17598, name, (class_1865<?>) type);

		return type;
	}

	public static RebornRecipeType<?> getRecipeType(class_2960 name){
		if(!recipeTypes.containsKey(name)){
			throw new RuntimeException("RebornRecipe type " + name + " not found");
		}
		return recipeTypes.get(name);
	}

	public static List<RebornRecipeType> getRecipeTypes(String namespace){
		return recipeTypes.values().stream().filter(rebornRecipeType -> rebornRecipeType.getName().method_12836().equals(namespace)).collect(Collectors.toList());
	}

	public static void validateRecipes(class_1937 world){
		//recipeTypes.forEach((key, value) -> validate(value, world));

		System.out.println("Validating recipes");
		world.method_8433().method_8127().forEach(identifier -> {
			try {
				class_1860 recipe =  world.method_8433().method_8130(identifier).get();
				class_1865 recipeSerializer = recipe.method_8119();
				class_2540 buf = new class_2540(Unpooled.buffer());
				recipeSerializer.method_8124(buf, recipe);

				class_1860 readback = recipeSerializer.method_8122(identifier, buf);
			} catch (Exception e){
				throw new RuntimeException("Failed to read " + identifier, e);
			}
		});
		System.out.println("Done");
	}

	private static <R extends RebornRecipe> void validate(RebornRecipeType<R> rebornRecipeType, class_1937 world){
		List<R> recipes = rebornRecipeType.getRecipes(world);

		for(RebornRecipe recipe1 : recipes){
			for(RebornRecipe recipe2 : recipes){
				if(recipe1 == recipe2){
					continue;
				}

				Validate.isTrue(recipe1.getRebornIngredients().size() > 0, recipe1.method_8114() + " has no inputs");
				Validate.isTrue(recipe2.getRebornIngredients().size() > 0, recipe2.method_8114() + " has no inputs");
				Validate.isTrue(recipe1.getOutputs().size() > 0, recipe1.method_8114() + " has no outputs");
				Validate.isTrue(recipe2.getOutputs().size() > 0, recipe2.method_8114() + " has no outputs");

				boolean hasAll = true;

				for(RebornIngredient recipe1Input : recipe1.getRebornIngredients()){
					boolean matches = false;
					for(class_1799 testStack : recipe1Input.getPreviewStacks()){
						for(RebornIngredient recipe2Input : recipe2.getRebornIngredients()){
							if(recipe2Input.test(testStack)){
								matches = true;
							}
						}
					}

					if(!matches){
						hasAll = false;
					}
				}

				if(hasAll){
					System.out.println(recipe1.method_8114() + " conflicts with " + recipe2.method_8114());
				}

			}
		}
	}


}
