/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.client.screen.builder;

import org.apache.commons.lang3.Range;
import org.apache.commons.lang3.tuple.MutableTriple;
import org.apache.commons.lang3.tuple.Pair;
import reborncore.common.blockentity.MachineBaseBlockEntity;
import reborncore.common.util.ItemUtils;
import reborncore.mixin.common.AccessorScreenHandler;

import java.util.ArrayList;
import java.util.List;
import java.util.function.*;
import net.minecraft.class_1263;
import net.minecraft.class_1657;
import net.minecraft.class_1703;
import net.minecraft.class_1712;
import net.minecraft.class_1715;
import net.minecraft.class_1735;
import net.minecraft.class_1799;

public class BuiltScreenHandler extends class_1703 implements ExtendedScreenHandlerListener {

	private final String name;

	private final Predicate<class_1657> canInteract;
	private final List<Range<Integer>> playerSlotRanges;
	private final List<Range<Integer>> blockEntitySlotRanges;

	private final ArrayList<MutableTriple<IntSupplier, IntConsumer, Short>> shortValues;
	private final ArrayList<MutableTriple<IntSupplier, IntConsumer, Integer>> integerValues;
	private final ArrayList<MutableTriple<Supplier, Consumer, Object>> objectValues;
	private List<Consumer<class_1715>> craftEvents;
	private Integer[] integerParts;

	private final MachineBaseBlockEntity blockEntity;

	public BuiltScreenHandler(int syncID, final String name, final Predicate<class_1657> canInteract,
							  final List<Range<Integer>> playerSlotRange,
							  final List<Range<Integer>> blockEntitySlotRange, MachineBaseBlockEntity blockEntity) {
		super(null, syncID);
		this.name = name;

		this.canInteract = canInteract;

		this.playerSlotRanges = playerSlotRange;
		this.blockEntitySlotRanges = blockEntitySlotRange;

		this.shortValues = new ArrayList<>();
		this.integerValues = new ArrayList<>();
		this.objectValues = new ArrayList<>();

		this.blockEntity = blockEntity;
	}

	public void addShortSync(final List<Pair<IntSupplier, IntConsumer>> syncables) {

		for (final Pair<IntSupplier, IntConsumer> syncable : syncables) {
			this.shortValues.add(MutableTriple.of(syncable.getLeft(), syncable.getRight(), (short) 0));
		}
		this.shortValues.trimToSize();
	}

	public void addIntegerSync(final List<Pair<IntSupplier, IntConsumer>> syncables) {

		for (final Pair<IntSupplier, IntConsumer> syncable : syncables) {
			this.integerValues.add(MutableTriple.of(syncable.getLeft(), syncable.getRight(), 0));
		}
		this.integerValues.trimToSize();
		this.integerParts = new Integer[this.integerValues.size()];
	}

	public void addObjectSync(final List<Pair<Supplier, Consumer>> syncables) {

		for (final Pair<Supplier, Consumer> syncable : syncables) {
			this.objectValues.add(MutableTriple.of(syncable.getLeft(), syncable.getRight(), null));
		}
		this.objectValues.trimToSize();
	}

	public void addCraftEvents(final List<Consumer<class_1715>> craftEvents) {
		this.craftEvents = craftEvents;
	}

	@Override
	public boolean method_7597(final class_1657 playerIn) {
		return this.canInteract.test(playerIn);
	}

	@Override
	public final void method_7609(final class_1263 inv) {
		if (!this.craftEvents.isEmpty()) {
			this.craftEvents.forEach(consumer -> consumer.accept((class_1715) inv));
		}
	}

	@Override
	public void method_7623() {
		super.method_7623();

		for (final class_1712 listener : ((AccessorScreenHandler)(this)).getListeners()) {

			int i = 0;
			if (!this.shortValues.isEmpty()) {
				for (final MutableTriple<IntSupplier, IntConsumer, Short> value : this.shortValues) {
					final short supplied = (short) value.getLeft().getAsInt();
					if (supplied != value.getRight()) {

						listener.method_7633(this, i, supplied);
						value.setRight(supplied);
					}
					i++;
				}
			}

			if (!this.integerValues.isEmpty()) {
				for (final MutableTriple<IntSupplier, IntConsumer, Integer> value : this.integerValues) {
					final int supplied = value.getLeft().getAsInt();
					if (supplied != value.getRight()) {

						listener.method_7633(this, i, supplied >> 16);
						listener.method_7633(this, i + 1, (short) (supplied & 0xFFFF));
						value.setRight(supplied);
					}
					i += 2;
				}
			}

			if (!this.objectValues.isEmpty()) {
				int objects = 0;
				for (final MutableTriple<Supplier, Consumer, Object> value : this.objectValues) {
					final Object supplied = value.getLeft().get();
					if(supplied != value.getRight()){
						sendObject(listener,this, objects, supplied);
						value.setRight(supplied);
					}
					objects++;
				}
			}
		}
	}

	@Override
	public void method_7596(final class_1712 listener) {
		super.method_7596(listener);

		int i = 0;
		if (!this.shortValues.isEmpty()) {
			for (final MutableTriple<IntSupplier, IntConsumer, Short> value : this.shortValues) {
				final short supplied = (short) value.getLeft().getAsInt();

				listener.method_7633(this, i, supplied);
				value.setRight(supplied);
				i++;
			}
		}

		if (!this.integerValues.isEmpty()) {
			for (final MutableTriple<IntSupplier, IntConsumer, Integer> value : this.integerValues) {
				final int supplied = value.getLeft().getAsInt();

				listener.method_7633(this, i, supplied >> 16);
				listener.method_7633(this, i + 1, (short) (supplied & 0xFFFF));
				value.setRight(supplied);
				i += 2;
			}
		}

		if (!this.objectValues.isEmpty()) {
			int objects = 0;
			for (final MutableTriple<Supplier, Consumer, Object> value : this.objectValues) {
				final Object supplied = value.getLeft();
				sendObject(listener, this, objects, ((Supplier) supplied).get());
				value.setRight(supplied);
				objects++;
			}
		}
	}

	@Override
	public void handleObject(int var, Object value) {
		this.objectValues.get(var).getMiddle().accept(value);
	}

	@Override
	public void method_7606(int id, int value) {
		if (id < this.shortValues.size()) {
			this.shortValues.get(id).getMiddle().accept((short) value);
			this.shortValues.get(id).setRight((short) value);
		} else if (id - this.shortValues.size() < this.integerValues.size() * 2) {

			if ((id - this.shortValues.size()) % 2 == 0) {
				this.integerParts[(id - this.shortValues.size()) / 2] = value;
			} else {
				this.integerValues.get((id - this.shortValues.size()) / 2).getMiddle().accept(
						(this.integerParts[(id - this.shortValues.size()) / 2] & 0xFFFF) << 16 | value & 0xFFFF);
			}
		}
	}

	@Override
	public class_1799 method_7601(final class_1657 player, final int index) {

		class_1799 originalStack = class_1799.field_8037;

		final class_1735 slot = this.field_7761.get(index);

		if (slot != null && slot.method_7681()) {

			final class_1799 stackInSlot = slot.method_7677();
			originalStack = stackInSlot.method_7972();

			boolean shifted = false;

			for (final Range<Integer> range : this.playerSlotRanges) {
				if (range.contains(index)) {

					if (this.shiftToBlockEntity(stackInSlot)) {
						shifted = true;
					}
					break;
				}
			}

			if (!shifted) {
				for (final Range<Integer> range : this.blockEntitySlotRanges) {
					if (range.contains(index)) {
						if (this.shiftToPlayer(stackInSlot)) {
							shifted = true;
						}
						break;
					}
				}
			}

			slot.method_7670(stackInSlot, originalStack);
			if (stackInSlot.method_7947() <= 0) {
				slot.method_7673(class_1799.field_8037);
			} else {
				slot.method_7668();
			}
			if (stackInSlot.method_7947() == originalStack.method_7947()) {
				return class_1799.field_8037;
			}
			slot.method_7667(player, stackInSlot);
		}
		return originalStack;

	}

	protected boolean shiftItemStack(final class_1799 stackToShift, final int start, final int end) {
		boolean changed = false;
		if (stackToShift.method_7946()) {
			for (int slotIndex = start; stackToShift.method_7947() > 0 && slotIndex < end; slotIndex++) {
				final class_1735 slot = this.field_7761.get(slotIndex);
				final class_1799 stackInSlot = slot.method_7677();
				if (!stackInSlot.method_7960() && ItemUtils.isItemEqual(stackInSlot, stackToShift, true, true)
					&& slot.method_7680(stackToShift)) {
					final int resultingStackSize = stackInSlot.method_7947() + stackToShift.method_7947();
					final int max = Math.min(stackToShift.method_7914(), slot.method_7675());
					if (resultingStackSize <= max) {
						stackToShift.method_7939(0);
						stackInSlot.method_7939(resultingStackSize);
						slot.method_7668();
						changed = true;
					} else if (stackInSlot.method_7947() < max) {
						stackToShift.method_7934(max - stackInSlot.method_7947());
						stackInSlot.method_7939(max);
						slot.method_7668();
						changed = true;
					}
				}
			}
		}
		if (stackToShift.method_7947() > 0) {
			for (int slotIndex = start; stackToShift.method_7947() > 0 && slotIndex < end; slotIndex++) {
				final class_1735 slot = this.field_7761.get(slotIndex);
				class_1799 stackInSlot = slot.method_7677();
				if (stackInSlot.method_7960() && slot.method_7680(stackToShift)) {
					final int max = Math.min(stackToShift.method_7914(), slot.method_7675());
					stackInSlot = stackToShift.method_7972();
					stackInSlot.method_7939(Math.min(stackToShift.method_7947(), max));
					stackToShift.method_7934(stackInSlot.method_7947());
					slot.method_7673(stackInSlot);
					slot.method_7668();
					changed = true;
				}
			}
		}
		return changed;
	}

	private boolean shiftToBlockEntity(final class_1799 stackToShift) {
		if(!blockEntity.getOptionalInventory().isPresent()){
			return false;
		}
		for (final Range<Integer> range : this.blockEntitySlotRanges) {
			if (this.shiftItemStack(stackToShift, range.getMinimum(), range.getMaximum() + 1)) {
				return true;
			}
		}
		return false;
	}

	private boolean shiftToPlayer(final class_1799 stackToShift) {
		for (final Range<Integer> range : this.playerSlotRanges) {
			if (this.shiftItemStack(stackToShift, range.getMinimum(), range.getMaximum() + 1)) {
				return true;
			}
		}
		return false;
	}

	public String getName() {
		return this.name;
	}

	@Override
	public class_1735 method_7621(class_1735 slotIn) {
		return super.method_7621(slotIn);
	}
}
