/*
 * This file is part of TechReborn, licensed under the MIT License (MIT).
 *
 * Copyright (c) 2020 TechReborn
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package reborncore.client.gui.guibuilder;

import com.google.common.collect.Lists;
import com.mojang.blaze3d.systems.RenderSystem;
import net.fabricmc.fabric.api.client.render.fluid.v1.FluidRenderHandlerRegistry;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.class_1058;
import net.minecraft.class_1059;
import net.minecraft.class_124;
import net.minecraft.class_1799;
import net.minecraft.class_2561;
import net.minecraft.class_2585;
import net.minecraft.class_2588;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_350;
import net.minecraft.class_3612;
import net.minecraft.class_437;
import net.minecraft.class_4587;
import reborncore.api.IListInfoProvider;
import reborncore.client.RenderUtil;
import reborncore.client.gui.builder.GuiBase;
import reborncore.client.gui.builder.slot.GuiTab;
import reborncore.common.fluid.FluidUtil;
import reborncore.common.fluid.FluidValue;
import reborncore.common.fluid.container.FluidInstance;
import reborncore.common.powerSystem.PowerSystem;
import reborncore.common.powerSystem.PowerSystem.EnergySystem;
import reborncore.common.util.StringUtils;

import java.text.Format;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * Created by Gigabit101 on 08/08/2016.
 */
public class GuiBuilder {
	public static final class_2960 defaultTextureSheet = new class_2960("reborncore", "textures/gui/guielements.png");
	private static final class_2561 SPACE_TEXT = new class_2585(" ");
	static class_2960 resourceLocation;

	public GuiBuilder() {
		GuiBuilder.resourceLocation = defaultTextureSheet;
	}

	public GuiBuilder(class_2960 resourceLocation) {
		GuiBuilder.resourceLocation = resourceLocation;
	}
	
	public class_2960 getResourceLocation() {
		return resourceLocation;
	}

	public void drawDefaultBackground(class_4587 matrixStack, class_437 gui, int x, int y, int width, int height) {
		RenderSystem.color4f(1.0F, 1.0F, 1.0F, 1.0F);
		class_310.method_1551().method_1531().method_22813(resourceLocation);
		gui.method_25302(matrixStack, x, y, 0, 0, width / 2, height / 2);
		gui.method_25302(matrixStack, x + width / 2, y, 150 - width / 2, 0, width / 2, height / 2);
		gui.method_25302(matrixStack, x, y + height / 2, 0, 150 - height / 2, width / 2, height / 2);
		gui.method_25302(matrixStack, x + width / 2, y + height / 2, 150 - width / 2, 150 - height / 2, width / 2,
			height / 2);
	}

	public void drawPlayerSlots(class_4587 matrixStack, class_437 gui, int posX, int posY, boolean center) {
		class_310.method_1551().method_1531().method_22813(resourceLocation);

		if (center) {
			posX -= 81;
		}

		for (int y = 0; y < 3; y++) {
			for (int x = 0; x < 9; x++) {
				gui.method_25302(matrixStack, posX + x * 18, posY + y * 18, 150, 0, 18, 18);
			}
		}

		for (int x = 0; x < 9; x++) {
			gui.method_25302(matrixStack, posX + x * 18, posY + 58, 150, 0, 18, 18);
		}
	}

	public void drawSlot(class_4587 matrixStack, class_437 gui, int posX, int posY) {
		class_310.method_1551().method_1531().method_22813(resourceLocation);
		gui.method_25302(matrixStack, posX, posY, 150, 0, 18, 18);
	}
	
	public void drawText(class_4587 matrixStack, GuiBase<?> gui, class_2561 text, int x, int y, int color) {
		gui.getTextRenderer().method_27528(matrixStack, text, x, y, color);
	}

	public void drawProgressBar(class_4587 matrixStack, GuiBase<?> gui, double progress, int x, int y) {
		gui.getMinecraft().method_1531().method_22813(resourceLocation);
		gui.method_25302(matrixStack, x, y, 150, 18, 22, 15);
		int j = (int) (progress);
		if (j > 0) {
			gui.method_25302(matrixStack, x, y, 150, 34, j + 1, 15);
		}
	}

	public void drawOutputSlot(class_4587 matrixStack, GuiBase<?> gui, int x, int y) {
		gui.getMinecraft().method_1531().method_22813(resourceLocation);
		gui.method_25302(matrixStack, x, y, 174, 0, 26, 26);
	}

	/**
	 * Draws button with JEI icon in the given coords.
	 *
	 * @param gui GuiBase GUI to draw on
	 * @param x int Top left corner where to place button
	 * @param y int Top left corner where to place button
	 * @param layer Layer Layer to draw on
	 */
	public void drawJEIButton(class_4587 matrixStack, GuiBase<?> gui, int x, int y, GuiBase.Layer layer) {
		if (gui.hideGuiElements()) return;
		if (FabricLoader.getInstance().isModLoaded("jei")) {
			if (layer == GuiBase.Layer.BACKGROUND) {
				x += gui.getGuiLeft();
				y += gui.getGuiTop();
			}
			gui.getMinecraft().method_1531().method_22813(resourceLocation);
			gui.method_25302(matrixStack, x, y, 202, 0, 12, 12);
		}
	}

	/**
	 * Draws lock button in either locked or unlocked state
	 *
	 * @param gui GuiBase GUI to draw on
	 * @param x int Top left corner where to place button
	 * @param y int Top left corner where to place button
	 * @param mouseX int Mouse cursor position to check for tooltip
	 * @param mouseY int Mouse cursor position to check for tooltip
	 * @param layer Layer Layer to draw on
	 * @param locked boolean Set to true if it is in locked state
	 */
	public void drawLockButton(class_4587 matrixStack, GuiBase<?> gui, int x, int y, int mouseX, int mouseY, GuiBase.Layer layer, boolean locked) {
		if (gui.hideGuiElements()) return;
		if (layer == GuiBase.Layer.BACKGROUND) {
			x += gui.getGuiLeft();
			y += gui.getGuiTop();
		}
		gui.getMinecraft().method_1531().method_22813(resourceLocation);
		gui.method_25302(matrixStack, x, y, 174, 26 + (locked ? 12 : 0), 20, 12);
		if (gui.isPointInRect(x, y, 20, 12, mouseX, mouseY)) {
			List<class_2561> list = new ArrayList<>();
			if (locked) {
				list.add(new class_2588("reborncore.gui.tooltip.unlock_items"));
			} else {
				list.add(new class_2588("reborncore.gui.tooltip.lock_items"));
			}
			RenderSystem.pushMatrix();
			gui.method_25417(matrixStack, list, mouseX, mouseY);
			RenderSystem.popMatrix();
		}
	}

	/**
	 * Draws hologram toggle button
	 *
	 * @param gui GuiBase GUI to draw on
	 * @param x int Top left corner where to place button
	 * @param y int Top left corner where to place button
	 * @param mouseX int Mouse cursor position to check for tooltip
	 * @param mouseY int Mouse cursor position to check for tooltip
	 * @param layer Layer Layer to draw on
	 */
	public void drawHologramButton(class_4587 matrixStack, GuiBase<?> gui, int x, int y, int mouseX, int mouseY, GuiBase.Layer layer) {
		if (gui.isTabOpen()) return;
		if (layer == GuiBase.Layer.BACKGROUND) {
			x += gui.getGuiLeft();
			y += gui.getGuiTop();
		}
		gui.getMinecraft().method_1531().method_22813(resourceLocation);
		if (gui.getMachine().renderMultiblock == null) {
			gui.method_25302(matrixStack, x, y, 174, 50, 20, 12);
		} else {
			gui.method_25302(matrixStack, x, y, 174, 62, 20, 12);
		}
		if (gui.isPointInRect(x, y, 20, 12, mouseX, mouseY)) {
			List<class_2561> list = new ArrayList<>();
			list.add(new class_2588("reborncore.gui.tooltip.hologram"));
			RenderSystem.pushMatrix();
			if (layer == GuiBase.Layer.FOREGROUND) {
				mouseX -= gui.getGuiLeft();
				mouseY -= gui.getGuiTop();
			}
			gui.method_25417(matrixStack, list, mouseX, mouseY);
			RenderSystem.popMatrix();
		}
	}

	/**
	 * Draws big horizontal bar for heat value
	 *
	 * @param gui GuiBase GUI to draw on
	 * @param x int Top left corner where to place bar
	 * @param y int Top left corner where to place bar
	 * @param value int Current heat value
	 * @param max int Maximum heat value
	 * @param layer Layer Layer to draw on
	 */
	public void drawBigHeatBar(class_4587 matrixStack, GuiBase<?> gui, int x, int y, int value, int max, GuiBase.Layer layer) {
		if (gui.hideGuiElements()) return;
		if (layer == GuiBase.Layer.BACKGROUND) {
			x += gui.getGuiLeft();
			y += gui.getGuiTop();
		}
		gui.getMinecraft().method_1531().method_22813(resourceLocation);
		gui.method_25302(matrixStack, x, y, 26, 218, 114, 18);
		if (value != 0) {
			int j = (int) ((double) value / (double) max * 106);
			if (j < 0) {
				j = 0;
			}
			gui.method_25302(matrixStack, x + 4, y + 4, 26, 246, j, 10);

			class_2561 text = new class_2585(String.valueOf(value))
					.method_10852(new class_2588("reborncore.gui.heat"));

			gui.drawCentredText(matrixStack, text, y + 5, 0xFFFFFF, layer);
		}
	}

	/**
	 * Draws big horizontal blue bar
	 *
	 * @param gui GuiBase GUI to draw on
	 * @param x int Top left corner where to place bar
	 * @param y int Top left corner where to place bar
	 * @param value int Current value
	 * @param max int Maximum value
	 * @param mouseX int Mouse cursor position to check for tooltip
	 * @param mouseY int Mouse cursor position to check for tooltip
	 * @param suffix String String to put on the bar and tooltip after percentage value
	 * @param line2 String String to put into tooltip as a second line
	 * @param format String Formatted value to put on the bar
	 * @param layer Layer Layer to draw on
	 */
	public void drawBigBlueBar(class_4587 matrixStack, GuiBase<?> gui, int x, int y, int value, int max, int mouseX, int mouseY, String suffix, class_2561 line2, String format, GuiBase.Layer layer) {
		if (gui.hideGuiElements()) return;
		if (layer == GuiBase.Layer.BACKGROUND) {
			x += gui.getGuiLeft();
			y += gui.getGuiTop();
		}
		gui.getMinecraft().method_1531().method_22813(resourceLocation);
		int j = (int) ((double) value / (double) max * 106);
		if (j < 0) {
			j = 0;
		}
		gui.method_25302(matrixStack, x + 4, y + 4, 0, 236, j, 10);
		if (!suffix.equals("")) {
			suffix = " " + suffix;
		}
		gui.drawCentredText(matrixStack, new class_2585(format).method_27693(suffix), y + 5, 0xFFFFFF, layer);
		if (gui.isPointInRect(x, y, 114, 18, mouseX, mouseY)) {
			int percentage = percentage(max, value);
			List<class_2561> list = new ArrayList<>();

			list.add(
					new class_2585(String.valueOf(value))
						.method_27692(class_124.field_1065)
						.method_27693("/")
						.method_27693(String.valueOf(max))
						.method_27693(suffix)
			);

			list.add(
					new class_2585(String.valueOf(percentage))
						.method_27692(StringUtils.getPercentageColour(percentage))
						.method_27693("%")
						.method_10852(
								new class_2588("reborncore.gui.tooltip.dsu_fullness")
								.method_27692(class_124.field_1080)
						)
			);

			list.add(line2);

			if (value > max) {
				list.add(
						new class_2585("Yo this is storing more than it should be able to")
							.method_27692(class_124.field_1080)
				);
				list.add(
						new class_2585("prolly a bug")
								.method_27692(class_124.field_1080)
				);
				list.add(
						new class_2585("pls report and tell how tf you did this")
								.method_27692(class_124.field_1080)
				);
			}
			if (layer == GuiBase.Layer.FOREGROUND) {
				mouseX -= gui.getGuiLeft();
				mouseY -= gui.getGuiTop();
			}
			gui.method_25417(matrixStack, list, mouseX, mouseY);
			RenderSystem.disableLighting();
			RenderSystem.color4f(1, 1, 1, 1);
		}
	}

	public void drawBigBlueBar(class_4587 matrixStack, GuiBase<?> gui, int x, int y, int value, int max, int mouseX, int mouseY, String suffix, GuiBase.Layer layer) {
		drawBigBlueBar(matrixStack, gui, x, y, value, max, mouseX, mouseY, suffix, class_2585.field_24366, Integer.toString(value), layer);

	}

	public void drawBigBlueBar(class_4587 matrixStack, GuiBase<?> gui, int x, int y, int value, int max, int mouseX, int mouseY, GuiBase.Layer layer) {
		drawBigBlueBar(matrixStack, gui, x, y, value, max, mouseX, mouseY, "", class_2585.field_24366, "", layer);
	}

	/**
	 * Shades GUI and draw gray bar on top of GUI
	 *
	 * @param gui GuiBase GUI to draw on
	 * @param layer Layer Layer to draw on
	 */
	public void drawMultiblockMissingBar(class_4587 matrixStack, GuiBase<?> gui, GuiBase.Layer layer) {
		if (gui.hideGuiElements()) return;
		int x = 0;
		int y = 4;
		if (layer == GuiBase.Layer.BACKGROUND) {
			x += gui.getGuiLeft();
			y += gui.getGuiTop();
		}
		RenderSystem.disableLighting();
		RenderSystem.enableDepthTest();
		RenderSystem.colorMask(true, true, true, false);
		RenderUtil.drawGradientRect(0, x, y, x + 176, y + 20, 0x000000, 0xC0000000);
		RenderUtil.drawGradientRect(0, x, y + 20, x + 176, y + 20 + 48, 0xC0000000, 0xC0000000);
		RenderUtil.drawGradientRect(0, x, y + 68, x + 176, y + 70 + 20, 0xC0000000, 0x00000000);
		RenderSystem.colorMask(true, true, true, true);
		RenderSystem.disableDepthTest();
		gui.drawCentredText(matrixStack, new class_2588("reborncore.gui.missingmultiblock"), 43, 0xFFFFFF, layer);
	}

	/**
	 * Draws upgrade slots on the left side of machine GUI. Draws on the background
	 * level.
	 *
	 * @param gui GuiBase GUI to draw on
	 * @param x int Top left corner where to place slots
	 * @param y int Top left corner where to place slots
	 */
	public void drawUpgrades(class_4587 matrixStack, GuiBase<?> gui, int x, int y) {
		gui.getMinecraft().method_1531().method_22813(resourceLocation);
		gui.method_25302(matrixStack, x, y, 217, 0, 24, 81);
	}

	/**
	 * Draws tab on the left side of machine GUI. Draws on the background level.
	 *
	 * @param gui GuiBase GUI to draw on
	 * @param x int Top left corner where to place tab
	 * @param y int Top left corner where to place tab
	 * @param stack ItemStack Item to show as tab icon
	 */
	public void drawSlotTab(class_4587 matrixStack, GuiBase<?> gui, int x, int y, class_1799 stack) {
		gui.getMinecraft().method_1531().method_22813(resourceLocation);
		gui.method_25302(matrixStack, x, y, 217, 82, 24, 24);
		gui.getMinecraft().method_1480().method_4023(stack, x + 5, y + 4);
	}


    /**
     * Draws Slot Configuration tips instead of player inventory
     * @param gui GuiBase GUI to draw on
     * @param x int Top left corner where to place tips list
     * @param y int Top left corner where to place tips list
     * @param mouseX int Mouse cursor position
     * @param mouseY int Mouse cursor position
     */
	public void drawSlotConfigTips(class_4587 matrixStack, GuiBase<?> gui, int x, int y, int mouseX, int mouseY, GuiTab guiTab) {
		List<class_2561> tips = guiTab.getTips().stream()
				.map(class_2588::new)
				.collect(Collectors.toList());

		TipsListWidget explanation = new TipsListWidget(gui, gui.getScreenWidth() - 14, 54, y, y + 76, 9 + 2, tips);
		explanation.method_25333(x - 81);
		explanation.method_25394(matrixStack, mouseX, mouseY, 1.0f);
		RenderSystem.color4f(1, 1, 1, 1);
	}


	private class TipsListWidget extends class_350<TipsListWidget.TipsListEntry> {

		public TipsListWidget(GuiBase<?> gui, int width, int height, int top, int bottom, int entryHeight, List<class_2561> tips) {
			super(gui.getMinecraft(), width, height, top, bottom, entryHeight);
			for (class_2561 tip : tips){
				this.method_25321(new TipsListEntry(tip));
			}
		}

		@Override
		public int method_25322() {
			return 162;
		}

		@Override
		protected void method_25325(class_4587 matrixStack) {

		}

		private class TipsListEntry extends class_350.class_351<TipsListWidget.TipsListEntry> {
			private class_2561 tip;

			public TipsListEntry(class_2561 tip){
				this.tip = tip;
			}

			@Override
			public void method_25343(class_4587 matrixStack, int index, int y, int x, int width, int height, int mouseX, int mouseY, boolean hovering, float delta) {
				class_310.method_1551().field_1772.method_1712(tip, x, y, width, 11184810);
			}
		}
	}

	/**
	 * Draws energy output value and icon
	 *
	 * @param gui GuiBase GUI to draw on
	 * @param x int Top left corner where to place energy output
	 * @param y int Top left corner where to place energy output
	 * @param maxOutput int Energy output value
	 * @param layer Layer Layer to draw on
	 */
	public void drawEnergyOutput(class_4587 matrixStack, GuiBase<?> gui, int x, int y, int maxOutput, GuiBase.Layer layer) {
		if (gui.hideGuiElements()) return;
		class_2561 text = new class_2585(PowerSystem.getLocaliszedPowerFormattedNoSuffix(maxOutput))
				.method_10852(SPACE_TEXT)
				.method_27693(PowerSystem.getDisplayPower().abbreviation)
				.method_27693("\t");

		int width = gui.getTextRenderer().method_27525(text);
		gui.drawText(matrixStack, text, x - width - 2, y + 5, 0, layer);
		if (layer == GuiBase.Layer.BACKGROUND) {
			x += gui.getGuiLeft();
			y += gui.getGuiTop();
		}
		gui.getMinecraft().method_1531().method_22813(resourceLocation);
		gui.method_25302(matrixStack, x, y, 150, 91, 16, 16);
	}

	/**
	 * Draws progress arrow in direction specified.
	 *
	 * @param gui GuiBase GUI to draw on
	 * @param progress int Current progress
	 * @param maxProgress int Maximum progress
	 * @param x int Top left corner where to place progress arrow
	 * @param y int Top left corner where to place progress arrow
	 * @param mouseX int Mouse cursor position to check for tooltip
	 * @param mouseY int Mouse cursor position to check for tooltip
	 * @param direction ProgressDirection Direction of progress arrow
	 * @param layer Layer Layer to draw on
	 */
	public void drawProgressBar(class_4587 matrixStack, GuiBase<?> gui, int progress, int maxProgress, int x, int y, int mouseX, int mouseY, ProgressDirection direction, GuiBase.Layer layer) {
		if (gui.hideGuiElements()) return;
		if (layer == GuiBase.Layer.BACKGROUND) {
			x += gui.getGuiLeft();
			y += gui.getGuiTop();
		}

		gui.getMinecraft().method_1531().method_22813(resourceLocation);
		gui.method_25302(matrixStack, x, y, direction.x, direction.y, direction.width, direction.height);
		int j = (int) ((double) progress / (double) maxProgress * 16);
		if (j < 0) {
			j = 0;
		}

		switch (direction) {
			case RIGHT:
				gui.method_25302(matrixStack, x, y, direction.xActive, direction.yActive, j, 10);
				break;
			case LEFT:
				gui.method_25302(matrixStack, x + 16 - j, y, direction.xActive + 16 - j, direction.yActive, j, 10);
				break;
			case UP:
				gui.method_25302(matrixStack, x, y + 16 - j, direction.xActive, direction.yActive + 16 - j, 10, j);
				break;
			case DOWN:
				gui.method_25302(matrixStack, x, y, direction.xActive, direction.yActive, 10, j);
				break;
			default:
				return;
		}

		if (gui.isPointInRect(x, y, direction.width, direction.height, mouseX, mouseY)) {
			int percentage = percentage(maxProgress, progress);
			List<class_2561> list = new ArrayList<>();
			list.add(
					new class_2585(String.valueOf(percentage))
					.method_27692(StringUtils.getPercentageColour(percentage))
					.method_27693("%")
			);
			if (layer == GuiBase.Layer.FOREGROUND) {
				mouseX -= gui.getGuiLeft();
				mouseY -= gui.getGuiTop();
			}
			gui.method_25417(matrixStack, list, mouseX, mouseY);
			RenderSystem.disableLighting();
			RenderSystem.color4f(1, 1, 1, 1);
		}
	}

	/**
	 * Draws multi-energy bar
	 *
	 * @param gui GuiBase GUI to draw on
	 * @param x int Top left corner where to place energy bar
	 * @param y int Top left corner where to place energy bar
	 * @param energyStored int Current amount of energy
	 * @param maxEnergyStored int Maximum amount of energy
	 * @param mouseX int Mouse cursor position to check for tooltip
	 * @param mouseY int Mouse cursor position to check for tooltip
	 * @param buttonID int Button ID used to switch energy systems
	 * @param layer Layer Layer to draw on
	 */
	public void drawMultiEnergyBar(class_4587 matrixStack, GuiBase<?> gui, int x, int y, int energyStored, int maxEnergyStored, int mouseX,
			int mouseY, int buttonID, GuiBase.Layer layer) {
		if (gui.hideGuiElements()) return;
		if (layer == GuiBase.Layer.BACKGROUND) {
			x += gui.getGuiLeft();
			y += gui.getGuiTop();
		}

		EnergySystem displayPower = PowerSystem.getDisplayPower();
		class_310.method_1551().method_1531().method_22813(resourceLocation);
		gui.method_25302(matrixStack, x, y, displayPower.xBar - 15, displayPower.yBar - 1, 14, 50);
		int draw = (int) ((double) energyStored / (double) maxEnergyStored * (48));
		if (energyStored > maxEnergyStored) {
			draw = 48;
		}
		gui.method_25302(matrixStack, x + 1, y + 49 - draw, displayPower.xBar, 48 + displayPower.yBar - draw, 12, draw);
		int percentage = percentage(maxEnergyStored, energyStored);
		if (gui.isPointInRect(x + 1, y + 1, 11, 48, mouseX, mouseY)) {
			List<class_2561> list = Lists.newArrayList();
			list.add(
					new class_2585(PowerSystem.getLocaliszedPowerFormattedNoSuffix(energyStored))
					.method_27692(class_124.field_1065)
					.method_27693("/")
					.method_27693(PowerSystem.getLocaliszedPowerFormattedNoSuffix(maxEnergyStored))
					.method_10852(SPACE_TEXT)
					.method_27693(displayPower.abbreviation)
			);

			list.add(
					StringUtils.getPercentageText(percentage)
					.method_10852(SPACE_TEXT)
					.method_10852(
							new class_2588("reborncore.gui.tooltip.power_charged")
							.method_27692(class_124.field_1080)
					)
			);

			if (gui.be instanceof IListInfoProvider) {
				if (class_437.method_25442()) {
					((IListInfoProvider) gui.be).addInfo(list, true, true);
				} else {
					list.add(class_2585.field_24366);

					list.add(
							new class_2585("Shift")
							.method_27692(class_124.field_1078)
							.method_10852(SPACE_TEXT)
							.method_27692(class_124.field_1080)
							.method_10852(new class_2588("reborncore.gui.tooltip.power_moreinfo"))
					);
				}
			}
			if (layer == GuiBase.Layer.FOREGROUND) {
				mouseX -= gui.getGuiLeft();
				mouseY -= gui.getGuiTop();
			}
			gui.method_25417(matrixStack, list, mouseX, mouseY);
			RenderSystem.disableLighting();
			RenderSystem.color4f(1, 1, 1, 1);
		}
	}

	/**
	 * Draws tank and fluid inside it
	 *
	 * @param gui GuiBase GUI to draw on
	 * @param x int Top left corner of tank
	 * @param y int Top left corner of tank
	 * @param mouseX int Mouse cursor position to check for tooltip
	 * @param mouseY int Mouse cursor position to check for tooltip
	 * @param fluid FluidStack Fluid to draw in tank
	 * @param maxCapacity int Maximum tank capacity
	 * @param isTankEmpty boolean True if tank is empty
	 * @param layer Layer Layer to draw on
	 */
	public void drawTank(class_4587 matrixStack, GuiBase<?> gui, int x, int y, int mouseX, int mouseY, FluidInstance fluid, FluidValue maxCapacity, boolean isTankEmpty, GuiBase.Layer layer) {
		if (gui.hideGuiElements()) return;
		if (layer == GuiBase.Layer.BACKGROUND) {
			x += gui.getGuiLeft();
			y += gui.getGuiTop();
		}

		int percentage = 0;
		FluidValue amount = FluidValue.EMPTY;
		if (!isTankEmpty) {
			amount = fluid.getAmount();
			percentage = percentage(maxCapacity.getRawValue(), amount.getRawValue());
		}
		gui.getMinecraft().method_1531().method_22813(resourceLocation);
		gui.method_25302(matrixStack, x, y, 194, 26, 22, 56);
		if (!isTankEmpty) {
			drawFluid(matrixStack, gui, fluid, x + 4, y + 4, 14, 48, maxCapacity.getRawValue());
		}
		gui.method_25302(matrixStack, x + 3, y + 3, 194, 82, 16, 50);

		if (gui.isPointInRect(x, y, 22, 56, mouseX, mouseY)) {
			List<class_2561> list = new ArrayList<>();
			if (isTankEmpty) {
				list.add(new class_2588("reborncore.gui.tooltip.tank_empty").method_27692(class_124.field_1065));
			} else {
				list.add(
						new class_2585(String.format("%s / %s", amount, maxCapacity))
							.method_27692(class_124.field_1065)
							.method_10852(SPACE_TEXT)
							.method_27693(FluidUtil.getFluidName(fluid))
				);
			}

			list.add(
					StringUtils.getPercentageText(percentage)
						.method_27692(class_124.field_1080)
						.method_10852(SPACE_TEXT)
						.method_10852(new class_2588("reborncore.gui.tooltip.tank_fullness"))
			);

			if (layer == GuiBase.Layer.FOREGROUND) {
				mouseX -= gui.getGuiLeft();
				mouseY -= gui.getGuiTop();
			}
			gui.method_25417(matrixStack, list, mouseX, mouseY);
			RenderSystem.disableLighting();
			RenderSystem.color4f(1, 1, 1, 1);
		}
	}

	/**
	 * Draws fluid in tank
	 *
	 * @param gui GuiBase GUI to draw on
	 * @param fluid FluidStack Fluid to draw
	 * @param x int Top left corner of fluid
	 * @param y int Top left corner of fluid
	 * @param width int Width of fluid to draw
	 * @param height int Height of fluid to draw
	 * @param maxCapacity int Maximum capacity of tank
	 */
	public void drawFluid(class_4587 matrixStack, GuiBase<?> gui, FluidInstance fluid, int x, int y, int width, int height, int maxCapacity) {
		if(fluid.getFluid() == class_3612.field_15906){
			return;
		}
		gui.getMinecraft().method_1531().method_22813(class_1059.field_5275);
		y += height;
		final class_1058 sprite = FluidRenderHandlerRegistry.INSTANCE.get(fluid.getFluid()).getFluidSprites(gui.getMachine().method_10997(), gui.getMachine().method_11016(), fluid.getFluid().method_15785())[0];
		int color = FluidRenderHandlerRegistry.INSTANCE.get(fluid.getFluid()).getFluidColor(gui.getMachine().method_10997(), gui.getMachine().method_11016(), fluid.getFluid().method_15785());

		final int drawHeight = (int) (fluid.getAmount().getRawValue() / (maxCapacity * 1F) * height);
		final int iconHeight = sprite.method_4595();
		int offsetHeight = drawHeight;

		RenderSystem.color3f((color >> 16 & 255) / 255.0F, (float) (color >> 8 & 255) / 255.0F, (float) (color & 255) / 255.0F);

		int iteration = 0;
		while (offsetHeight != 0) {
			final int curHeight = offsetHeight < iconHeight ? offsetHeight : iconHeight;

			class_332.method_25298(matrixStack, x, y - offsetHeight, 0,  width, curHeight, sprite);
			offsetHeight -= curHeight;
			iteration++;
			if (iteration > 50) {
				break;
			}
		}
		RenderSystem.color3f(1F, 1F, 1F);

		gui.getMinecraft().method_1531().method_22813(resourceLocation);
	}

	/**
	 * Draws burning progress, similar to vanilla furnace
	 *
	 * @param gui GuiBase GUI to draw on
	 * @param progress int Current progress
	 * @param maxProgress int Maximum progress
	 * @param x int Top left corner where to place burn bar
	 * @param y int Top left corner where to place burn bar
	 * @param mouseX int Mouse cursor position to check for tooltip
	 * @param mouseY int Mouse cursor position to check for tooltip
	 * @param layer Layer Layer to draw on
	 */
	public void drawBurnBar(class_4587 matrixStack, GuiBase<?> gui, int progress, int maxProgress, int x, int y, int mouseX, int mouseY, GuiBase.Layer layer) {
		if (gui.hideGuiElements()) return;
		if (layer == GuiBase.Layer.BACKGROUND) {
			x += gui.getGuiLeft();
			y += gui.getGuiTop();
		}
		gui.getMinecraft().method_1531().method_22813(resourceLocation);
		gui.method_25302(matrixStack, x, y, 150, 64, 13, 13);
		int j = 13 - (int) ((double) progress / (double) maxProgress * 13);
		if (j > 0) {
			gui.method_25302(matrixStack, x, y + j, 150, 51 + j, 13, 13 - j);

		}
		if (gui.isPointInRect(x, y, 12, 12, mouseX, mouseY)) {
			int percentage = percentage(maxProgress, progress);
			List<class_2561> list = new ArrayList<>();
			list.add(StringUtils.getPercentageText(percentage));
			if (layer == GuiBase.Layer.FOREGROUND) {
				mouseX -= gui.getGuiLeft();
				mouseY -= gui.getGuiTop();
			}
			gui.method_25417(matrixStack, list, mouseX, mouseY);
			RenderSystem.disableLighting();
			RenderSystem.color4f(1, 1, 1, 1);
		}
	}

	/**
	 * Draws bar containing output slots
	 *
	 * @param gui GuiBase GUI to draw on
	 * @param x int Top left corner where to place slots bar
	 * @param y int Top left corner where to place slots bar
	 * @param count int Number of output slots
	 */
	public void drawOutputSlotBar(class_4587 matrixStack, GuiBase<?> gui, int x, int y, int count) {
		class_310.method_1551().method_1531().method_22813(resourceLocation);
		gui.method_25302(matrixStack, x, y, 150, 122, 3, 26);
		x += 3;
		for (int i = 1; i <= count; i++) {
			gui.method_25302(matrixStack, x, y, 150 + 3, 122, 20, 26);
			x += 20;
		}
		gui.method_25302(matrixStack, x, y, 150 + 23, 122, 3, 26);
	}

	protected int percentage(int MaxValue, int CurrentValue) {
		if (CurrentValue == 0) {
			return 0;
		}
		return (int) ((CurrentValue * 100.0f) / MaxValue);
	}

	public enum ProgressDirection {
		RIGHT(58, 150, 74, 150, 16, 10),
		LEFT(74, 160, 58, 160, 16, 10),
		DOWN(78, 170, 88, 170, 10, 16),
		UP(58, 170, 68, 170, 10, 16);
		public int x;
		public int y;
		public int xActive;
		public int yActive;
		public int width;
		public int height;

		ProgressDirection(int x, int y, int xActive, int yActive, int width, int height) {
			this.x = x;
			this.y = y;
			this.xActive = xActive;
			this.yActive = yActive;
			this.width = width;
			this.height = height;
		}
	}
}
